from __future__ import annotations

import difflib
import threading
from collections import defaultdict
from functools import lru_cache
from pathlib import Path
from typing import Dict, Iterable, List, Optional, Protocol, Set, Tuple

import enchant
from PyQt6.QtCore import QObject, pyqtSignal

from models.ProjectConfigModel import ProjectConfigModel, ProjectsConfigModel
from services.LanguageService import LanguageService
from utils.logger import log_exception, log_info
from utils.resource_path import resource_path

DEFAULT_DICTIONARY = "en_US"
CUSTOM_WORDS_LIMIT = 1000
MAX_SUGGESTIONS = 8
_MAX_CACHE_SIZE = 4096


class SpellcheckDict(Protocol):
    def check(self, word: str) -> bool:
        ...

    def suggest(self, word: str) -> List[str]:
        ...


_BROKER = enchant.Broker()
try:  # HunSpell is optional on some platforms (not available on minimal wheels)
    from enchant import HunSpell  # type: ignore
except ImportError:  # pragma: no cover - depends on pyenchant build
    HunSpell = None  # type: ignore


def _dictionaries_root() -> Path:
    return Path(resource_path("assets")) / "dictionaries"


def _configure_broker():
    dictionaries_root = _dictionaries_root()
    if not dictionaries_root.exists():
        log_info("Spellcheck dictionaries directory not found; using pyenchant defaults")
        return
    dictionaries_path = str(dictionaries_root.resolve())
    try:
        params = (
            ("enchant.myspell.dictionary.path", dictionaries_path),
            ("enchant.myspell.dictionary.path_default", dictionaries_path),
            ("enchant.hunspell.dictionary.path", dictionaries_path),
            ("enchant.hunspell.dictionary.path_default", dictionaries_path),
        )
        for param, value in params:
            _BROKER.set_param(param, value)
        log_info(f"Configured enchant broker dictionaries at {dictionaries_path}")
    except AttributeError:
        log_info("Enchant broker does not support set_param; relying on default dictionary lookup")
    except Exception:  # pragma: no cover - defensive guard
        log_exception("Failed to configure enchant broker", exc_info=True)


_configure_broker()


def _dictionary_code_variants(dictionary_code: Optional[str]) -> List[str]:
    variants: List[str] = []
    if dictionary_code:
        variants.append(dictionary_code)
    normalized = _normalize_locale_code(dictionary_code)
    if normalized and normalized not in variants:
        variants.append(normalized)
    if normalized:
        language = normalized.split('_', 1)[0]
        if language and language not in variants:
            variants.append(language)
    return variants


def _find_local_dictionary_paths(dictionary_code: Optional[str]) -> List[Tuple[str, Path, Path]]:
    dictionaries_root = _dictionaries_root()
    if not dictionaries_root.exists():
        return []
    candidates: List[Tuple[str, Path, Path]] = []
    for code in _dictionary_code_variants(dictionary_code):
        aff_path = dictionaries_root / f"{code}.aff"
        dic_path = dictionaries_root / f"{code}.dic"
        if aff_path.exists() and dic_path.exists():
            candidates.append((code, aff_path, dic_path))
    return candidates


def _load_hunspell_dict(dictionary_code: Optional[str]):
    if not HunSpell:
        return None
    for code, aff_path, dic_path in _find_local_dictionary_paths(dictionary_code):
        try:
            dictionary = HunSpell(str(aff_path), str(dic_path))
            log_info(f"Loaded Hunspell dictionary '{code}' from local assets")
            return dictionary
        except Exception:  # pragma: no cover - depends on environment
            log_exception(
                f"Failed to load Hunspell dictionary '{code}' from {aff_path} / {dic_path}",
                exc_info=True,
            )
    return None


def _load_word_list_dict(dictionary_code: Optional[str]) -> Optional[SpellcheckDict]:
    for code, _aff_path, dic_path in _find_local_dictionary_paths(dictionary_code):
        words = _load_word_list(dic_path)
        if words:
            log_info(f"Loaded fallback spellcheck dictionary '{code}' from local word list")
            return _WordListDictionary(code, words)
    return None


def _load_word_list(dic_path: Path) -> List[str]:
    if not dic_path.exists():
        return []
    words: Set[str] = set()
    try:
        with dic_path.open("r", encoding="utf-8", errors="ignore") as handle:
            for index, raw_line in enumerate(handle):
                line = raw_line.strip()
                if not line:
                    continue
                if index == 0 and line.isdigit():
                    # Hunspell dictionaries may list entry counts on the first line
                    continue
                word = line.split("/", 1)[0].strip()
                if word:
                    words.add(word.lower())
    except Exception:  # pragma: no cover - filesystem/encoding edge cases
        log_exception(f"Unable to read dictionary file {dic_path}", exc_info=True)
        return []
    return sorted(words)


def _request_broker_dict(dictionary_code: str) -> Optional[enchant.Dict]:
    try:
        if hasattr(_BROKER, "dict_exists") and _BROKER.dict_exists(dictionary_code):
            log_info(f"Loading spellcheck dictionary '{dictionary_code}' via broker")
            return _BROKER.request_dict(dictionary_code)
    except enchant.errors.DictNotFoundError:
        return None
    except Exception:  # pragma: no cover - defensive guard
        log_exception("Enchant broker request failed", exc_info=True)
    return None


class _WordListDictionary:
    def __init__(self, code: str, words: List[str]):
        self.code = code
        self._words_set = set(words)
        self._words_list = words

    def check(self, word: str) -> bool:
        return word.lower() in self._words_set

    def suggest(self, word: str) -> List[str]:
        lowered = word.lower()
        return difflib.get_close_matches(lowered, self._words_list, n=MAX_SUGGESTIONS, cutoff=0.7)


@lru_cache(maxsize=8)
def _load_enchant_dict(dictionary_code: str) -> SpellcheckDict:
    dictionary = _request_broker_dict(dictionary_code)
    if dictionary:
        return dictionary
    hunspell_dict = _load_hunspell_dict(dictionary_code)
    if hunspell_dict:
        return hunspell_dict
    word_list_dict = _load_word_list_dict(dictionary_code)
    if word_list_dict:
        return word_list_dict
    try:
        log_info(f"Falling back to default enchant dictionary for '{dictionary_code}'")
        return enchant.Dict(dictionary_code)
    except enchant.errors.DictNotFoundError:
        log_exception(f"Spellcheck dictionary '{dictionary_code}' not found", exc_info=True)
        return enchant.Dict(DEFAULT_DICTIONARY)


def _try_load_optional_dict(dictionary_code: str) -> Optional[SpellcheckDict]:
    if not dictionary_code:
        return None
    dictionary = _request_broker_dict(dictionary_code)
    if dictionary:
        return dictionary
    hunspell_dict = _load_hunspell_dict(dictionary_code)
    if hunspell_dict:
        return hunspell_dict
    word_list_dict = _load_word_list_dict(dictionary_code)
    if word_list_dict:
        return word_list_dict
    try:
        return enchant.Dict(dictionary_code)
    except enchant.errors.DictNotFoundError:
        log_info(f"Spellcheck dictionary '{dictionary_code}' not found; skipping")
        return None


def _normalize_locale_code(locale: Optional[str]) -> Optional[str]:
    if not locale:
        return None
    sanitized = locale.replace('-', '_').strip()
    if not sanitized:
        return None
    parts = sanitized.split('_', 1)
    language = parts[0].lower()
    if len(parts) == 1:
        return language
    region = parts[1].upper()
    return f"{language}_{region}"


def _candidate_locale_codes(locale: Optional[str]) -> List[str]:
    normalized = _normalize_locale_code(locale)
    if not normalized:
        return []
    candidates = [normalized]
    language = normalized.split('_', 1)[0]
    if language and language not in candidates:
        candidates.append(language)
    return candidates


class SpellcheckService(QObject):
    customWordAdded = pyqtSignal(str)
    customWordRemoved = pyqtSignal(str)

    def __init__(self, parent=None, projects_config: ProjectsConfigModel | None = None):
        super().__init__(parent)
        self._lock = threading.RLock()
        self._projects_config = projects_config or ProjectsConfigModel.load()
        self._ignore_session: Dict[str | None, Set[str]] = defaultdict(set)
        self._dictionary: SpellcheckDict = _load_enchant_dict(DEFAULT_DICTIONARY)
        self._ui_dictionary: Optional[SpellcheckDict] = None
        self._ui_dictionary_code: Optional[str] = None
        self._dictionary_namespace = DEFAULT_DICTIONARY
        self._check_cache: Dict[str, bool] = {}
        self._suggestions_cache: Dict[str, List[str]] = {}
        self._refresh_dictionaries()

    def reload(self, projects_config: ProjectsConfigModel | None = None):
        with self._lock:
            if projects_config is not None:
                self._projects_config = projects_config
            else:
                self._projects_config = ProjectsConfigModel.load()
            self._refresh_dictionaries()
            self._clear_caches()

    def _refresh_dictionaries(self):
        self._dictionary = _load_enchant_dict(DEFAULT_DICTIONARY)
        self._ui_dictionary_code, self._ui_dictionary = self._load_ui_dictionary()
        parts = [DEFAULT_DICTIONARY]
        if self._ui_dictionary_code and self._ui_dictionary_code != DEFAULT_DICTIONARY:
            parts.append(self._ui_dictionary_code)
        self._dictionary_namespace = "|".join(parts)

    def _load_ui_dictionary(self) -> Tuple[Optional[str], Optional[SpellcheckDict]]:
        locale = None
        try:
            locale = LanguageService.currentLocale()
        except Exception:  # pragma: no cover - defensive guard
            log_exception("Unable to determine UI locale for spellcheck", exc_info=True)
        for code in _candidate_locale_codes(locale):
            if not code or code.lower() == DEFAULT_DICTIONARY.lower():
                continue
            dictionary = _try_load_optional_dict(code)
            if dictionary:
                log_info(f"Using UI spellcheck dictionary '{code}' in addition to English")
                return code, dictionary
        if locale:
            log_info(f"UI locale '{locale}' has no dedicated spellcheck dictionary; using English only")
        return None, None

    def _cache_key(self, project_id: Optional[str], word: str) -> str:
        return f"{project_id or '__global__'}::{self._dictionary_namespace}::{word}"

    def _suggestions_cache_key(self, word: str) -> str:
        return f"{self._dictionary_namespace}::{word}"

    def _clear_caches(self):
        self._check_cache.clear()
        self._suggestions_cache.clear()

    def is_enabled(self, project_id: Optional[str]) -> bool:
        if not project_id:
            return False
        project = self._get_project_config(project_id)
        return project.spellcheck_enabled is not False

    def set_enabled(self, project_id: str, enabled: bool):
        with self._lock:
            project = self._get_project_config(project_id)
            project.spellcheck_enabled = enabled
            self._projects_config.save()
            self._clear_caches()

    def ignore_for_session(self, word: str, project_id: Optional[str] = None):
        normalized = word.strip().lower()
        if not normalized:
            return
        key = project_id or "__global__"
        self._ignore_session[key].add(normalized)
        self._check_cache.pop(self._cache_key(project_id, normalized), None)

    def reset_session_ignores(self, project_id: Optional[str] = None):
        if project_id is None:
            self._ignore_session.clear()
        else:
            key = project_id or "__global__"
            self._ignore_session.pop(key, None)

    def get_custom_words(self, project_id: Optional[str]) -> List[str]:
        if not project_id:
            return []
        project = self._get_project_config(project_id)
        return project.spellcheck_custom_words or []

    def add_custom_word(self, project_id: Optional[str], word: str) -> bool:
        if not project_id:
            return False
        normalized = word.strip().lower()
        if not normalized:
            return False
        with self._lock:
            project = self._get_project_config(project_id)
            words = project.spellcheck_custom_words or []
            if normalized in words:
                return False
            if len(words) >= CUSTOM_WORDS_LIMIT:
                return False
            words.append(normalized)
            project.spellcheck_custom_words = words
            self._projects_config.save()
            self._clear_caches()
            self.customWordAdded.emit(normalized)
        return True

    def remove_custom_word(self, project_id: Optional[str], word: str) -> bool:
        if not project_id:
            return False
        normalized = word.strip().lower()
        if not normalized:
            return False
        with self._lock:
            project = self._get_project_config(project_id)
            words = project.spellcheck_custom_words or []
            if normalized not in words:
                return False
            words.remove(normalized)
            project.spellcheck_custom_words = words
            self._projects_config.save()
            self._clear_caches()
            self.customWordRemoved.emit(normalized)
        return True

    def check_word(self, word: str, project_id: Optional[str]) -> bool:
        normalized = word.strip().lower()
        if not normalized:
            return True
        key = project_id or "__global__"
        if normalized in self._ignore_session.get(key, set()):
            return True
        cache_key = self._cache_key(project_id, normalized)
        cached = self._check_cache.get(cache_key)
        if cached is not None:
            return cached
        custom_words = set(self.get_custom_words(project_id))
        if normalized in custom_words:
            self._check_cache[cache_key] = True
            return True
        result = self._dictionary.check(normalized)
        if not result and self._ui_dictionary:
            result = self._ui_dictionary.check(normalized)
        if len(self._check_cache) >= _MAX_CACHE_SIZE:
            self._check_cache.clear()
        self._check_cache[cache_key] = result
        return result

    def suggestions(self, word: str, limit: int = MAX_SUGGESTIONS) -> List[str]:
        normalized = word.strip().lower()
        if not normalized:
            return []
        cache_key = self._suggestions_cache_key(normalized)
        cached = self._suggestions_cache.get(cache_key)
        if cached is not None:
            return cached[:limit]
        suggestions: List[str] = []
        seen: Set[str] = set()
        for dictionary in filter(None, (self._dictionary, self._ui_dictionary)):
            for suggestion in dictionary.suggest(normalized):
                if suggestion in seen:
                    continue
                seen.add(suggestion)
                suggestions.append(suggestion)
                if len(suggestions) >= MAX_SUGGESTIONS:
                    break
            if len(suggestions) >= MAX_SUGGESTIONS:
                break
        if len(self._suggestions_cache) >= _MAX_CACHE_SIZE:
            self._suggestions_cache.clear()
        self._suggestions_cache[cache_key] = suggestions
        return suggestions[:limit]

    def _get_project_config(self, project_id: str) -> ProjectConfigModel:
        if project_id not in self._projects_config.projects:
            self._projects_config.projects[project_id] = ProjectConfigModel(
                id=project_id,
                path=None,
                skip_large_project_warning=None,
                bash_show_console=True,
            )
        return self._projects_config.projects[project_id]

    def update_custom_words(self, project_id: Optional[str], words: Iterable[str]):
        if not project_id:
            return
        sanitized = []
        for word in words:
            normalized = word.strip().lower()
            if normalized:
                sanitized.append(normalized)
        with self._lock:
            project = self._get_project_config(project_id)
            project.spellcheck_custom_words = sanitized[:CUSTOM_WORDS_LIMIT]
            self._projects_config.save()
            self._clear_caches()

    def get_misspelled_words(self, text: str, project_id: Optional[str]) -> List[str]:
        unique_words = {word.strip(".,!?;:\"'()[]{}"): word for word in text.split()}
        misspelled = []
        for normalized, original in unique_words.items():
            if not self.check_word(normalized, project_id):
                misspelled.append(original)
        return misspelled
