import os

from PyQt6.QtCore import QLocale
from babel import Locale, UnknownLocaleError

from core.I18n import i18n, _
from models.config.AppConfigModel import AppConfigModel
from ui.components.StandardDialog import StandardDialog
from utils import resource_path
from utils.platform_helper import is_windows


class LanguageService:
    @staticmethod
    def __defaultLocale(fallback="en_US") -> Locale:
        qt_name = QLocale.system().name().replace("-", "_") or fallback
        try:
            return Locale.parse(qt_name)
        except UnknownLocaleError:
            lang = qt_name.split("_", 1)[0]
            try:
                return Locale.parse(lang)
            except UnknownLocaleError:
                return Locale.parse(fallback)

    @staticmethod
    def initService():
        print(f"System locale {str(LanguageService.__defaultLocale())}")
        code = LanguageService.currentLocale()
        print(f"Current locale is {code}")
        i18n.set_language(code)

    @staticmethod
    def currentLocale() -> str:
        locale = AppConfigModel.load().language
        if not locale:
            locale = str(LanguageService.__defaultLocale())
        return locale

    @staticmethod
    def currentLanguage() -> str:
        locale = Locale.parse(LanguageService.currentLocale())
        return locale.language_name

    @staticmethod
    def getListLanguages() -> list[Locale]:
        path = resource_path("assets/locales")
        folders = [entry.name for entry in os.scandir(path) if entry.is_dir()]
        languages = [Locale.parse(x) for x in folders]
        return languages

    @staticmethod
    def setLanguage(locale: Locale, parent = None) -> None:
        code = str(locale)
        current = LanguageService.currentLocale()
        if code != current:
            i18n.set_language(code)
            AppConfigModel.load().language = code
            AppConfigModel.load().save()
            from services.ApplicationService import ApplicationService
            if not is_windows():
                ApplicationService.restart()
            else:
                StandardDialog.info(parent, _("Notification"), _("The language will be changed after restart."))
