import json
from enum import Enum
from typing import List

from .filesystem_actions import FileSystemActions
from .bash_runner import BashRunner
from .return_action import ReturnAction


class ProjectRunner:
    class Function(Enum):
        WRITE_FILE = 'write_file'
        CREATE_DIR = 'create_dir'
        DELETE = 'delete'
        MOVE_PROJECT_FILE = 'move_project_file'
        RUN_BASH_SCRIPT = 'run_bash_script'

    @staticmethod
    def run(fn, project_dir) -> List[ReturnAction]:
        function_name = fn.get('name')
        args = json.loads(fn.get('arguments', '{}'))
        path = FileSystemActions.clear_path(args.get('path'))

        try:
            function = ProjectRunner.Function(function_name)
        except ValueError:
            return []

        match function:
            case ProjectRunner.Function.WRITE_FILE:
                return FileSystemActions.write_file(project_dir, path, args.get('body'))
            case ProjectRunner.Function.CREATE_DIR:
                return FileSystemActions.create_dir(project_dir, path)
            case ProjectRunner.Function.DELETE:
                return FileSystemActions.delete_path(project_dir, path)
            case ProjectRunner.Function.MOVE_PROJECT_FILE:
                new_path = FileSystemActions.clear_path(args.get('new_path'))
                return FileSystemActions.move_file(project_dir, path, new_path)
            case ProjectRunner.Function.RUN_BASH_SCRIPT:
                return BashRunner.run_script(
                    project_dir,
                    args.get('script'),
                    timeout_seconds=args.get('timeout_seconds'),
                    shell=args.get('shell'),
                    show_console=args.get('show_console', False)
                )
        return []
