import os
import shutil
from typing import List, Tuple

from utils.project_info import clean_string
from .return_action import ReturnAction


class FileSystemActions:
    @staticmethod
    def clear_path(path: str | None) -> str | None:
        """Legacy helper kept for compatibility (strips [ROOT]/ prefix)."""
        if path is None:
            return None
        if path.startswith("[ROOT]/"):
            return path.replace("[ROOT]/", "", 1)
        return path

    @staticmethod
    def _resolve_project_path(project_dir: str, path: str | None) -> Tuple[str, str]:
        """Validate provided path and return (relative_path, absolute_path)."""
        if path is None:
            raise ValueError("path is required")

        cleaned = FileSystemActions.clear_path(path.strip())
        if not cleaned:
            raise ValueError("path cannot be empty")
        if os.path.isabs(cleaned):
            raise ValueError("absolute paths are not allowed")

        normalized_relative = os.path.normpath(cleaned)
        if normalized_relative in ("", "."):
            raise ValueError("path must point to a file or directory inside the project")

        abs_project_dir = os.path.abspath(project_dir)
        abs_target = os.path.abspath(os.path.join(abs_project_dir, normalized_relative))

        try:
            common_root = os.path.commonpath([abs_project_dir, abs_target])
        except ValueError as exc:  # Different drives on Windows
            raise ValueError("path escapes the project directory") from exc

        if common_root != abs_project_dir:
            raise ValueError("path escapes the project directory")

        return normalized_relative.replace("\\", "/"), abs_target

    @staticmethod
    def create_dir(project_dir: str, path: str) -> List[ReturnAction]:
        try:
            rel_path, abs_path = FileSystemActions._resolve_project_path(project_dir, path)
        except ValueError as exc:
            return [ReturnAction(f"Invalid directory path '{path}': {exc}")]

        if os.path.exists(abs_path):
            log = f"Directory {rel_path} already exists!"
        else:
            os.makedirs(abs_path)
            log = f"Directory {rel_path} created successfully!"
        return [ReturnAction(log)]

    @staticmethod
    def write_file(project_dir: str, path: str, body: str | None) -> List[ReturnAction]:
        try:
            rel_path, file_path = FileSystemActions._resolve_project_path(project_dir, path)
        except ValueError as exc:
            return [ReturnAction(f"Invalid file path '{path}': {exc}")]

        content = clean_string(body or "")
        lines_in_old_file = 0
        if os.path.exists(file_path):
            with open(file_path, "r", encoding="utf-8") as existing:
                lines_in_old_file = sum(1 for _ in existing)

        lines_in_new_file = sum(1 for _ in content.splitlines())
        os.makedirs(os.path.dirname(file_path), exist_ok=True)

        with open(file_path, "w", encoding="utf-8") as dst:
            dst.write(content)

        log = (
            f"File {rel_path} created or updated successfully! "
            f"Was {lines_in_old_file} lines before, now {lines_in_new_file} lines."
        )
        extension = os.path.splitext(rel_path)[1]
        return [
            ReturnAction(log),
            ReturnAction(extension, type=ReturnAction.ReturnActionType.ADD_EXTENSION),
        ]

    @staticmethod
    def delete_path(project_dir: str, path: str) -> List[ReturnAction]:
        try:
            rel_path, abs_path = FileSystemActions._resolve_project_path(project_dir, path)
        except ValueError as exc:
            return [ReturnAction(f"Invalid delete path '{path}': {exc}")]

        if not os.path.exists(abs_path):
            return [ReturnAction(f"Path {rel_path} does not exist!")]

        if os.path.isfile(abs_path):
            os.remove(abs_path)
            log = f"File {rel_path} deleted successfully!"
        else:
            shutil.rmtree(abs_path, ignore_errors=True)
            log = f"Directory {rel_path} deleted successfully!"
        return [ReturnAction(log)]

    @staticmethod
    def move_file(project_dir: str, path: str, new_path: str) -> List[ReturnAction]:
        try:
            rel_old_path, abs_old_path = FileSystemActions._resolve_project_path(project_dir, path)
            rel_new_path, abs_new_path = FileSystemActions._resolve_project_path(project_dir, new_path)
        except ValueError as exc:
            return [ReturnAction(f"Invalid move path: {exc}")]

        if not os.path.exists(abs_old_path):
            return [ReturnAction(f"Path {rel_old_path} does not exist!")]
        if os.path.exists(abs_new_path):
            return [ReturnAction(f"Path {rel_new_path} already exists!")]

        os.makedirs(os.path.dirname(abs_new_path), exist_ok=True)
        shutil.move(abs_old_path, abs_new_path)
        return [ReturnAction(f"File {rel_old_path} moved successfully to {rel_new_path}!")]
