import os
import shutil
import subprocess
import tempfile
from dataclasses import dataclass
from typing import List, Optional

from utils.project_info import clean_string
from .return_action import ReturnAction


@dataclass
class BashExecutionResult:
    exit_code: int
    stdout: str
    stderr: str


class BashRunner:
    MIN_TIMEOUT = 10
    MAX_TIMEOUT = 300
    DEFAULT_TIMEOUT = 120

    _WIN_BASH_CANDIDATES = [
        r"C:\Program Files\Git\usr\bin\bash.exe",
        r"C:\Program Files\Git\bin\bash.exe",
        r"C:\Program Files (x86)\Git\usr\bin\bash.exe",
        r"C:\Program Files (x86)\Git\bin\bash.exe",
    ]

    _NIX_BASH_CANDIDATES = [
        "/bin/bash",
        "/usr/bin/bash",
        "/bin/sh",
        "/usr/bin/sh",
    ]

    @classmethod
    def detect_bash(cls, preferred: str | None = None) -> Optional[str]:
        candidates: List[str] = []

        if os.name == "nt":
            candidates.extend(cls._WIN_BASH_CANDIDATES)

        if preferred:
            candidates.append(preferred)

        env_shell = os.environ.get("SHELL")
        if env_shell:
            candidates.append(env_shell)

        if os.name != "nt":
            candidates.extend(cls._NIX_BASH_CANDIDATES)

        candidates.extend(["bash", "sh"] if os.name != "nt" else ["bash"])

        for cand in candidates:
            if not cand:
                continue
            cand = cand.strip().strip('"')
            if not cand:
                continue
            if os.path.isabs(cand) and os.path.exists(cand):
                return cand
            resolved = shutil.which(cand)
            if resolved:
                return resolved
        return None

    @staticmethod
    def _write_tmp_script(project_dir: str, content: str) -> str:
        with tempfile.NamedTemporaryFile(
            mode="w",
            delete=False,
            suffix=".sh",
            dir=project_dir,
            encoding="utf-8",
            newline="\n",
        ) as tmp:
            tmp.write("#!/usr/bin/env bash\n")
            tmp.write("set -e\n")
            tmp.write("set -o pipefail\n\n")
            tmp.write(content)
            if not content.endswith("\n"):
                tmp.write("\n")
            path = tmp.name

        try:
            os.chmod(path, 0o700)
        except OSError:
            pass
        return path

    @staticmethod
    def _cleanup(paths: List[str | None]) -> None:
        for p in paths:
            if p and os.path.exists(p):
                try:
                    os.remove(p)
                except OSError:
                    pass

    @staticmethod
    def _to_bash_path(path: str) -> str:
        """
        Convert Windows path -> Git Bash path.
        Uses cygpath if available, otherwise /c/... fallback.
        """
        if os.name != "nt":
            return path

        path = os.path.abspath(path)

        cygpath = r"C:\Program Files\Git\usr\bin\cygpath.exe"
        if os.path.exists(cygpath):
            try:
                r = subprocess.run([cygpath, "-u", path], capture_output=True, text=True, check=False)
                out = (r.stdout or "").strip()
                if r.returncode == 0 and out:
                    return out
            except OSError:
                pass

        drive, rest = os.path.splitdrive(path)
        rest = rest.replace("\\", "/")
        drive_letter = drive.rstrip(":").lower()
        if drive_letter:
            if not rest.startswith("/"):
                rest = "/" + rest
            return f"/{drive_letter}{rest}"
        return rest

    @classmethod
    def _build_cmd(cls, bash_path: str, script_path: str, project_dir: str) -> List[str]:
        """
        Predictable execution:
        - Windows: bash -lc 'cd "<dir>" && bash "<script>"'
        - Unix:    bash "<script>"
        """
        if os.name != "nt":
            return [bash_path, script_path]

        workdir_bash = cls._to_bash_path(project_dir)
        script_bash = cls._to_bash_path(script_path)

        cmd_str = f'cd "{workdir_bash}" && bash "{script_bash}"'
        return [bash_path, "-lc", cmd_str]

    @classmethod
    def _execute(
        cls,
        bash_path: str,
        script_path: str,
        project_dir: str,
        timeout: int,
        show_console: bool,
    ) -> BashExecutionResult:
        cmd = cls._build_cmd(bash_path, script_path, project_dir)

        creationflags = 0
        if os.name == "nt" and show_console:
            creationflags |= getattr(subprocess, "CREATE_NEW_CONSOLE", 0)

        completed = subprocess.run(
            cmd,
            cwd=project_dir,
            capture_output=True,
            text=True,
            timeout=timeout,
            check=False,
            creationflags=creationflags,
        )
        return BashExecutionResult(
            exit_code=completed.returncode,
            stdout=(completed.stdout or "").strip(),
            stderr=(completed.stderr or "").strip(),
        )

    @classmethod
    def run_script(
        cls,
        project_dir: str,
        script: str,
        timeout_seconds: Optional[int],
        shell: Optional[str],
        show_console: bool,
    ) -> List[ReturnAction]:
        script_content = clean_string(script or "")
        if not script_content.strip():
            return [ReturnAction("run_bash_script: script is empty, nothing to execute.")]

        bash_path = cls.detect_bash(shell)
        if not bash_path:
            return [ReturnAction("run_bash_script: bash not found. Install Git Bash or provide shell path.")]

        timeout = cls.DEFAULT_TIMEOUT
        if timeout_seconds is not None:
            try:
                timeout = max(cls.MIN_TIMEOUT, min(cls.MAX_TIMEOUT, int(timeout_seconds)))
            except (ValueError, TypeError):
                timeout = cls.DEFAULT_TIMEOUT

        logs: List[ReturnAction] = []
        script_path: Optional[str] = None

        try:
            script_path = cls._write_tmp_script(project_dir, script_content)
            logs.append(ReturnAction(f"Executing bash script via: {bash_path} (timeout {timeout}s)"))

            result = cls._execute(
                bash_path=bash_path,
                script_path=script_path,
                project_dir=project_dir,
                timeout=timeout,
                show_console=show_console,
            )

            logs.append(ReturnAction(f"Exit code: {result.exit_code}"))
            if result.stdout:
                logs.append(ReturnAction(f"STDOUT:\n{result.stdout}"))
            if result.stderr:
                logs.append(ReturnAction(f"STDERR:\n{result.stderr}"))

        except subprocess.TimeoutExpired:
            logs.append(ReturnAction(f"run_bash_script timed out after {timeout}s."))
        except FileNotFoundError as exc:
            logs.append(ReturnAction(f"run_bash_script failed: {exc}"))
        except Exception as exc:
            logs.append(ReturnAction(f"run_bash_script error: {exc}"))
        finally:
            cls._cleanup([script_path])

        return logs
