from __future__ import annotations

from dataclasses import dataclass
from typing import Any

from core.I18n import _
from core.ProjectRunner import ReturnAction
from core.focus.focus_manager import FocusManager
from core.formChat.project_manager import ProjectManager


@dataclass
class ToolExecutionSummary:
    stdout: list[str]
    stderr: list[str]
    status: list[str]
    extensions: list[str]
    focus_updates: list[str]

    def to_message(self) -> str:
        sections: list[str] = []
        if self.status:
            sections.extend(self.status)
        if self.focus_updates:
            sections.append("\n".join(self.focus_updates))
        stdout_block = ToolOutputFormatter.format_shell_block(_("STDOUT"), "\n".join(self.stdout)) if self.stdout else ""
        stderr_block = ToolOutputFormatter.format_shell_block(_("STDERR"), "\n".join(self.stderr)) if self.stderr else ""
        for block in (stdout_block, stderr_block):
            if block:
                sections.append(block)
        return "\n\n".join(sections) if sections else _("Command finished with no output.")


class ToolOutputFormatter:
    FOCUS_TOOL_PREFIX = "FOCUS_UPDATE"

    @staticmethod
    async def collect_results(tool_results: list[ReturnAction], project_manager: ProjectManager, project_id: str) -> ToolExecutionSummary:
        summary = ToolExecutionSummary(stdout=[], stderr=[], status=[], extensions=[], focus_updates=[])

        for result in tool_results:
            if result.type == ReturnAction.ReturnActionType.LOG and result.value:
                normalized = result.value.strip()
                if normalized.startswith(ToolOutputFormatter.FOCUS_TOOL_PREFIX):
                    summary.focus_updates.append(ToolOutputFormatter._format_focus_section(normalized))
                    continue
                label, payload = ToolOutputFormatter._split_prefix(normalized)
                if label == "STDOUT":
                    summary.stdout.append(payload)
                elif label == "STDERR":
                    summary.stderr.append(payload)
                else:
                    summary.status.append(normalized)
            elif result.type == ReturnAction.ReturnActionType.ADD_EXTENSION and result.value:
                summary.extensions.append(result.value)

        if summary.extensions:
            await project_manager.add_extensions(summary.extensions, project_id=project_id)

        return summary

    @staticmethod
    def _split_prefix(message: str) -> tuple[str, str]:
        if ":" not in message:
            return message.upper(), ""
        prefix, rest = message.split(":", 1)
        return prefix.strip().upper(), rest.lstrip("\n")

    @staticmethod
    def format_shell_block(label: str, text: str | None) -> str:
        if not text:
            return ""
        normalized = "\n".join(line.rstrip() for line in text.split("\n"))
        return (f"{label}:\n```"
                f"{normalized}"
                f"```")

    @staticmethod
    def _format_focus_section(message: str) -> str:
        # Expected format: "FOCUS_UPDATE:<json>" or "FOCUS_UPDATE:<plain message>"
        _, _, payload = message.partition(":")
        payload = payload.strip()
        if payload and payload.startswith("{"):
            try:
                import json

                data: dict[str, Any] = json.loads(payload)
                title = data.get("title", "Unnamed focus")
                summary = data.get("summary", "")
                plan_preview = data.get("plan_preview")
                progress = data.get("progress")
                lines = [_("Focus update:"), f"• {title}"]
                if plan_preview:
                    lines.append(plan_preview)
                if progress:
                    lines.append(progress)
                if summary:
                    lines.append(summary)
                return "\n".join(lines)
            except Exception:
                pass
        return f"Focus update: {payload or _('No details provided.')}"
