from __future__ import annotations

import json
from dataclasses import dataclass
from typing import Any

from core.I18n import _
from core.focus.focus_manager import FocusManager
from core.formChat.project_manager import ProjectManager


@dataclass
class NormalizedMessage:
    message_type: str
    role: str
    message: str
    attachments: list[dict[str, Any]] | None
    project_id: str
    project_dir: str | None
    function_call: dict[str, Any] | None


class MessageNormalizer:
    def __init__(self, project_manager: ProjectManager, focus_manager: FocusManager | None = None):
        self._project_manager = project_manager
        self._focus_manager = focus_manager or FocusManager.instance()

    def normalize(self, msg: dict[str, Any], default_project_id: str) -> NormalizedMessage:
        message_type = msg.get("message_type", "message")
        project_id = msg.get("project_id", default_project_id)
        project_dir = self._project_manager.get_project_path(project_id)
        attachments = msg.get("attachments") if isinstance(msg.get("attachments"), list) and msg["attachments"] else None
        role = msg.get("role", "assistant")
        content = msg.get("message", "")
        function_call: dict[str, Any] | None = None
        raw_data = msg.get("message_raw_data")

        if message_type == "function_call":
            function_call = self._parse_function_call(raw_data)
        elif message_type == "function_call_output":
            content = self._summarize_function_call_output(raw_data, content, project_id)
            role = "sys tool"

        return NormalizedMessage(
            message_type=message_type,
            role=role,
            message=content,
            attachments=attachments,
            project_id=project_id,
            project_dir=project_dir,
            function_call=function_call,
        )

    @staticmethod
    def _parse_function_call(raw_data: Any) -> dict[str, Any] | None:
        if raw_data is None:
            return None
        if isinstance(raw_data, dict):
            return raw_data
        if isinstance(raw_data, str):
            try:
                return json.loads(raw_data)
            except json.JSONDecodeError:
                return None
        return None

    def _summarize_function_call_output(self, raw_data: Any, fallback_content: str, project_id: str | None) -> str:
        tool_call = self._parse_function_call(raw_data)
        if not tool_call:
            return fallback_content

        tool_name = tool_call.get("name")
        fn_args = self._safe_parse_arguments(tool_call)

        if tool_name == "get_project_file":
            return self._summarize_get_project_file(fn_args, fallback_content)

        if tool_name == "search_in_files":
            return self._summarize_search_in_files(fn_args, fallback_content)

        if tool_name and tool_name.startswith("focus_"):
            focus_result = self._summarize_focus_tool(tool_name, fallback_content)
            if focus_result and project_id:
                self._focus_manager.trigger_immediate_refresh(project_id)
            if focus_result:
                return focus_result

        return fallback_content

    @staticmethod
    def _safe_parse_arguments(tool_call: dict[str, Any]) -> dict[str, Any]:
        arguments_raw = tool_call.get("arguments")
        if not arguments_raw:
            return {}
        if isinstance(arguments_raw, dict):
            return arguments_raw
        if isinstance(arguments_raw, str):
            try:
                return json.loads(arguments_raw)
            except json.JSONDecodeError:
                return {}
        return {}

    def _summarize_get_project_file(self, fn_args: dict[str, Any], fallback_content: str) -> str:
        paths = self._collect_paths(fn_args)
        if not paths:
            sanitized_path = _("unknown path")
            return _(
                "File retrieved successfully: {path}. Contents hidden per policy."
            ).format(path=sanitized_path)

        if len(paths) == 1:
            path = paths[0]
            line_count = len(fallback_content.splitlines()) if fallback_content else 0
            char_count = len(fallback_content or "")
            return (
                _("File retrieved successfully: {path}.").format(path=path)
                + " "
                + _("{lines} line(s), {chars} character(s). Contents hidden per policy.").format(
                    lines=line_count,
                    chars=char_count,
                )
            )

        payload = self._parse_multi_file_payload(fallback_content)
        successes: list[str] = []
        missing: list[str] = []
        for entry in payload:
            entry_path = entry.get("path") or _("unknown path")
            if entry.get("status") == "ok" and isinstance(entry.get("body"), str):
                body = entry["body"]
                line_count = len(body.splitlines()) if body else 0
                char_count = len(body)
                successes.append(f"{entry_path} ({line_count} line(s), {char_count} char(s))")
            else:
                missing.append(entry_path)

        total_requested = len(paths)
        success_count = len(successes)
        parts = [
            _("Files retrieved: {success}/{total}. Contents hidden per policy.").format(
                success=success_count,
                total=total_requested,
            )
        ]
        if successes:
            parts.append(_("Opened: {details}.").format(details=", ".join(successes)))
        if missing or success_count < total_requested:
            unresolved = missing or [p for p in paths if p not in {e.get("path") for e in payload}]
            if unresolved:
                parts.append(_("Missing/failed: {items}.").format(items=", ".join(unresolved)))
        return " " + " ".join(parts)

    def _summarize_search_in_files(self, fn_args: dict[str, Any], fallback_content: str) -> str:
        queries = self._collect_queries(fn_args)
        matches = self._group_matches_by_query(fallback_content)
        fallback_total = sum(matches.values())

        if not queries:
            return _("Search completed. Matches found: {count}.").format(count=fallback_total)

        if len(queries) == 1:
            query = queries[0]
            match_count = matches.get(query, fallback_total)
            return _("Search completed for '{query}'. Matches found: {count}.").format(
                query=query,
                count=match_count,
            )

        lines = [_("Search completed for multiple queries:")]
        for query in queries:
            count = matches.get(query, 0)
            lines.append(f"• {query}: {count}")
        return "\n".join(lines)

    @staticmethod
    def _parse_multi_file_payload(fallback_content: str) -> list[dict[str, Any]]:
        try:
            payload = json.loads(fallback_content)
        except (json.JSONDecodeError, TypeError):
            return []
        if isinstance(payload, list):
            return [entry for entry in payload if isinstance(entry, dict)]
        return []

    @staticmethod
    def _collect_paths(fn_args: dict[str, Any]) -> list[str]:
        paths: list[str] = []
        seen: set[str] = set()

        def _push(value: Any):
            if not isinstance(value, str):
                return
            normalized = value.replace("[ROOT]", "").lstrip("/")
            if not normalized:
                return
            if normalized in seen:
                return
            paths.append(normalized)
            seen.add(normalized)

        _push(fn_args.get("path"))
        extra = fn_args.get("paths") or []
        if isinstance(extra, list):
            for item in extra:
                _push(item)
        return paths

    @staticmethod
    def _collect_queries(fn_args: dict[str, Any]) -> list[str]:
        queries: list[str] = []
        seen: set[str] = set()

        def _push(value: Any):
            if not isinstance(value, str):
                return
            candidate = value.strip()
            if not candidate or candidate in seen:
                return
            queries.append(candidate)
            seen.add(candidate)

        _push(fn_args.get("query"))
        extra = fn_args.get("queries") or []
        if isinstance(extra, list):
            for item in extra:
                _push(item)
        return queries

    @staticmethod
    def _group_matches_by_query(fallback_content: str) -> dict[str, int]:
        matches: dict[str, int] = {}
        try:
            payload = json.loads(fallback_content)
        except (json.JSONDecodeError, TypeError):
            return matches
        if not isinstance(payload, list):
            return matches
        for entry in payload:
            if not isinstance(entry, dict):
                continue
            query = entry.get("query")
            if not isinstance(query, str):
                continue
            matches[query] = matches.get(query, 0) + 1
        return matches

    @staticmethod
    def _extract_match_count(content: str) -> int:
        if not content:
            return 0
        try:
            parsed = json.loads(content)
            if isinstance(parsed, list):
                return len(parsed)
        except json.JSONDecodeError:
            return 0
        return 0

    @staticmethod
    def _summarize_focus_tool(tool_name: str, fallback_content: str) -> str | None:
        try:
            payload = json.loads(fallback_content)
        except Exception:
            payload = None

        if tool_name == "focus_list" and isinstance(payload, dict):
            count = len(payload.get("focuses", []))
            active = payload.get("active_focus_id")
            if active:
                return _("Focus list refreshed. Active focus ID: {active}, total focuses: {count}.").format(
                    active=active,
                    count=count,
                )
            return _("Focus list refreshed. Total focuses: {count}.").format(count=count)

        if tool_name == "focus_get_active_summary" and isinstance(payload, dict):
            title = payload.get("title") or _("No active focus")
            progress = payload.get("progress") or _("Progress unknown")
            return _("Active focus summary: {title}. {progress}").format(title=title, progress=progress)

        if tool_name in {"focus_create", "focus_update", "focus_set_active"}:
            if isinstance(payload, dict):
                title = payload.get("title") or payload.get("id") or _("Unnamed focus")
                status = payload.get("status") or _("Focus updated.")
            else:
                title = _("Unnamed focus")
                status = fallback_content or _("Action completed.")
            return _("{status} Target: {title}.").format(status=status, title=title)

        return None
