from __future__ import annotations

import asyncio
import traceback
from dataclasses import dataclass
from typing import Any, TYPE_CHECKING

from PyQt6.QtGui import QTextCursor

import api.client as client
from api.client import InfoAPIException
from core.I18n import _
from core.ProjectRunner import ProjectRunner, ReturnAction
from core.SoundManager import SoundManager
from core.formChat.project_manager import ProjectManager, normalize_project_path
from core.message_handler.bash_consent import BashConsentManager
from core.message_handler.frontend_tools import FrontendToolRunner
from core.message_handler.message_normalizer import MessageNormalizer
from core.message_handler.tool_output_formatter import ToolOutputFormatter

TECHNICAL_CONTINUE_MARKER = "[~continue~]"

if TYPE_CHECKING:  # pragma: no cover
    from ui.formChat.ChatView import ChatView


@dataclass
class SendContext:
    message: str
    attachments: list[str]
    project_id: str
    display_user_message: bool
    is_technical_continue: bool
    use_saved_structure: bool

    @property
    def outbound_text(self) -> str:
        return TECHNICAL_CONTINUE_MARKER if self.is_technical_continue else self.message

    @property
    def should_display_user(self) -> bool:
        return (
            self.display_user_message
            and not self.is_technical_continue
            and (self.message or self.attachments)
        )


class MessageHandler:
    def __init__(self, chat_view: ChatView):
        self.chat_view = chat_view
        self._bash_consent_manager = BashConsentManager(chat_view)
        self._frontend_tool_runner = FrontendToolRunner(chat_view, self._bash_consent_manager)
        self._project_manager = ProjectManager.instance()
        self._message_normalizer = MessageNormalizer(self._project_manager)
        self._last_sent_user_message: str | None = None
        self._active_sync_cancel_event: asyncio.Event | None = None
        self._active_sync_project_id: str | None = None
        self._spellcheck_service = self._project_manager.spellcheck_service
        self.commands = {
            "pl": client.project_list,
            "pc": client.project_create,
        }

    async def process_message(self):
        message = self.chat_view.message_input_view.message_input.toPlainText().strip()
        self.chat_view.message_input_view.message_input.clear()

        images_base64: list[str] = []
        if hasattr(self.chat_view, "attachments_widget"):
            images_base64 = self.chat_view.attachments_widget.get_images_base64()
            self.chat_view.attachments_widget.clear_images()

        if message.startswith("!"):
            command_line = message[1:]
            command = command_line.split(" ")[0]
            params_line = command_line[len(command):].strip()
            params = params_line.split("|") if params_line else []
            await self.system_command(command, params)
        elif message or images_base64:
            await self.send_message(message, images_base64)

    async def system_command(self, command, params):
        project_id = self._project_manager.project_id
        if command not in self.commands:
            self.chat_view.append_message("sys", f"{_('Unknown command')}: {command}", project_id=project_id)
            return
        try:
            response = await self.commands[command](*params)
            self.chat_view.append_message("sys", str(response), project_id=project_id)
        except Exception as exc:  # pragma: no cover
            self.chat_view.append_message("sys", str(exc), project_id=project_id)

    async def send_message(
        self,
        message: str,
        attachments: list[str] | None = None,
        project_id: str | None = None,
        display_user_message: bool = True,
        is_technical_continue: bool = False,
        use_saved_structure: bool = False,
    ):
        self.chat_view.hide_autocomplete_inline()
        context = self._create_send_context(
            message,
            attachments or [],
            project_id,
            display_user_message,
            is_technical_continue,
            use_saved_structure,
        )
        if context is None:
            return

        attachments_payload = self._build_attachments_payload(context.attachments) if context.attachments else None

        self._render_outbound_if_needed(context)
        self.chat_view.show_thinking()

        try:
            project_structure = None
            if not context.use_saved_structure:
                project_structure = await self._prepare_project_for_send(context.project_id)
                if project_structure is None:
                    return
            else:
                if not self._ensure_project_directory(context.project_id):
                    return

            if not context.is_technical_continue:
                self._last_sent_user_message = context.message

            response = await client.send_message(
                context.outbound_text,
                context.project_id,
                project_structure,
                attachments_payload,
                use_saved_structure=context.use_saved_structure,
            )
            self._persist_spellcheck_context()
            follow_up_triggered = await self._handle_response(response, context.project_id)
            if not follow_up_triggered:
                self.chat_view.hide_thinking()
            if not context.is_technical_continue:
                self._last_sent_user_message = None
                SoundManager.instance().play(SoundManager.Sound.EAGLE)
        except InfoAPIException as info_exc:
            self.chat_view.append_message("info", str(info_exc.detail), project_id=context.project_id)
            self._restore_input(message)
            self.chat_view.hide_thinking()
        except Exception as exc:  # pragma: no cover
            self.chat_view.append_message("Error", _("Failed to send message, please try again."), project_id=context.project_id)
            print(f"Error sending message: {exc}")
            print(traceback.format_exc())
            self._restore_input(message)
            self.chat_view.hide_thinking()

    def _persist_spellcheck_context(self):
        project_id = self._project_manager.project_id
        if not project_id:
            return
        text = self.chat_view.message_input_view.message_input.toPlainText()
        words = text.split()
        ignores = [word for word in words if word and not self._spellcheck_service.check_word(word, project_id)]
        if ignores:
            self._spellcheck_service.update_custom_words(project_id, ignores)

    def _create_send_context(
        self,
        message: str,
        attachments: list[str],
        project_id: str | None,
        display_user_message: bool,
        is_technical_continue: bool,
        use_saved_structure: bool,
    ) -> SendContext | None:
        resolved_project_id = project_id or self._project_manager.project_id
        if not resolved_project_id:
            return None
        return SendContext(
            message=message,
            attachments=attachments,
            project_id=resolved_project_id,
            display_user_message=display_user_message,
            is_technical_continue=is_technical_continue,
            use_saved_structure=use_saved_structure,
        )

    async def _prepare_project_for_send(self, project_id: str) -> dict[str, Any] | None:
        if not self._ensure_project_directory(project_id):
            return None
        return await self._sync_project_structure(project_id)

    def _render_outbound_if_needed(self, context: SendContext):
        if not context.should_display_user:
            return
        attachments_payload = context.attachments if context.attachments else None
        self.chat_view.append_message(
            "user",
            context.message,
            attachments=attachments_payload,
            project_id=context.project_id,
        )

    async def _handle_response(self, response: dict[str, Any], default_project_id: str) -> bool:
        messages = response.get("messages", [])
        normalized_messages = []
        for msg in messages:
            if msg.get("role") == "user" and self._is_duplicate_user_message(msg):
                continue
            normalized_messages.append(self._message_normalizer.normalize(msg, default_project_id))

        tool_results = {}
        run_bash_calls = []

        for normalized in normalized_messages:
            if self._is_tool_call(normalized):
                self._collect_tool_call(normalized, tool_results, run_bash_calls)
            else:
                self._render_normalized_message(normalized)

        if run_bash_calls:
            bash_results = await self._frontend_tool_runner.handle_run_bash_calls(run_bash_calls)
            for project_id, results in bash_results:
                if results:
                    tool_results.setdefault(project_id, []).extend(results)

        aggregated_messages = await self._summarize_tool_results(tool_results)
        follow_up_triggered = await self._dispatch_follow_ups(aggregated_messages, response, default_project_id)
        self._render_quick_replies(response, default_project_id)
        return follow_up_triggered

    def _is_tool_call(self, normalized_message):
        return normalized_message.message_type == "function_call" and bool(normalized_message.function_call)

    def _collect_tool_call(self, normalized_message, tool_results, run_bash_calls):
        function_call = normalized_message.function_call or {}
        if function_call.get("name") == "run_bash_script":
            run_bash_calls.append(
                {
                    "fn": function_call,
                    "project_dir": normalized_message.project_dir,
                    "project_id": normalized_message.project_id,
                }
            )
            return
        tool_output = ProjectRunner.run(function_call, normalized_message.project_dir)
        if tool_output:
            tool_results.setdefault(normalized_message.project_id, []).extend(tool_output)

    def _render_normalized_message(self, normalized_message):
        self.chat_view.append_message_with_project(
            normalized_message.role,
            normalized_message.message,
            attachments=normalized_message.attachments,
            message_type=normalized_message.message_type,
            project_id=normalized_message.project_id,
        )

    async def _summarize_tool_results(self, tool_results: dict[str, list[ReturnAction]]):
        aggregated_messages: dict[str, str] = {}
        for project_id, results in tool_results.items():
            summary = await ToolOutputFormatter.collect_results(results, self._project_manager, project_id)
            final_message = summary.to_message()
            if not final_message:
                continue
            aggregated_messages[project_id] = final_message
            self.chat_view.append_message("sys tool", final_message, project_id=project_id)
        return aggregated_messages

    async def _dispatch_follow_ups(
        self,
        aggregated_messages: dict[str, str],
        response: dict[str, Any],
        default_project_id: str,
    ) -> bool:
        follow_up_triggered = False
        for project_id, message_text in aggregated_messages.items():
            follow_up_triggered = True
            await self.send_message(
                message_text,
                project_id=project_id,
                display_user_message=False,
                is_technical_continue=False
            )

        if self._should_auto_continue(response, aggregated_messages):
            follow_up_triggered = True
            await self.send_message(
                TECHNICAL_CONTINUE_MARKER,
                project_id=default_project_id,
                display_user_message=False,
                is_technical_continue=True,
            )
        return follow_up_triggered

    def _should_auto_continue(self, response: dict[str, Any], aggregated_messages: dict[str, str]) -> bool:
        if aggregated_messages:
            return False
        messages = response.get("messages") or []
        if not messages:
            return False
        last_message = messages[-1]
        return last_message.get("message_type") in {"function_call", "function_call_output"}

    def _render_quick_replies(self, response: dict[str, Any], default_project_id: str):
        quick_replies = response.get("user_quick_replies") or []
        if not quick_replies:
            return
        replies_html = "\n".join(
            f"<p onclick=\"triggerAction('quick_reply', '{reply}')\">{reply}</p>" for reply in quick_replies
        )
        last_msg_project_id = response.get("messages", [{}])[-1].get("project_id", default_project_id)
        self.chat_view.append_message("sys tmp quick_replies", replies_html, False, project_id=last_msg_project_id)

    def cancel_active_sync(self):
        if self._active_sync_project_id and self._active_sync_cancel_event and not self._active_sync_cancel_event.is_set():
            self._active_sync_cancel_event.set()

    def _ensure_project_directory(self, project_id: str) -> bool:
        project_dir = self._project_manager.get_project_path(project_id)
        if project_dir:
            return True
        selected_dir = self.chat_view.request_directory_from_user()
        if not selected_dir:
            return False
        normalized_path = normalize_project_path(selected_dir)
        self._project_manager.set_project_info_path(project_id, normalized_path)
        return True

    async def _sync_project_structure(self, project_id: str) -> dict[str, Any] | None:
        project_manager = self._project_manager
        indicator_visible = {"active": False}
        cancel_event = asyncio.Event()
        self._active_sync_cancel_event = cancel_event
        self._active_sync_project_id = project_id

        def progress_callback(msg: str):
            if project_manager.project_id != project_id:
                return
            indicator_visible["active"] = True
            self.chat_view.show_sync_status(msg)

        try:
            project_structure = await project_manager.sync_project_structure(
                progress_callback=progress_callback,
                project_id=project_id,
                cancel_event=cancel_event,
            )
        finally:
            self._active_sync_project_id = None
            self._active_sync_cancel_event = None
            if indicator_visible["active"]:
                self.chat_view.hide_sync_status()

        if project_structure is None and project_manager.project_id == project_id:
            self.chat_view.append_message("sys", _("Project structure sending cancelled."), project_id=project_id)
        return project_structure

    def _render_user_message(self, message: str, attachments: list[str], project_id: str):
        attachments_payload = attachments if attachments else None
        self.chat_view.append_message(
            "user",
            message,
            attachments=attachments_payload,
            project_id=project_id,
        )

    def _restore_input(self, message: str):
        self.chat_view.message_input_view.message_input.setPlainText(message)
        self.chat_view.message_input_view.message_input.moveCursor(QTextCursor.MoveOperation.End)

    def _is_duplicate_user_message(self, msg: dict[str, Any]) -> bool:
        incoming = (msg.get("message") or "").strip()
        last_sent = (self._last_sent_user_message or "").strip()
        return bool(last_sent and incoming == last_sent)

    @staticmethod
    def _build_attachments_payload(attachments: list[str]) -> list[dict[str, Any]]:
        return [
            {
                "name": f"image_{index}.png",
                "type": "image",
                "content_base64": "data:image/png;base64," + attachment,
            }
            for index, attachment in enumerate(attachments)
        ]

    async def load_history(self):
        self.chat_view.clear_chat_display()
        project_id = self._project_manager.project_id
        if not project_id:
            return
        try:
            response = await client.get_messages(project_id=project_id, limit=10, offset=0)
            for message in reversed(response):
                normalized = self._message_normalizer.normalize(message, project_id)
                self.chat_view.append_message_with_project(
                    normalized.role,
                    normalized.message,
                    attachments=normalized.attachments,
                    message_type=normalized.message_type,
                    project_id=normalized.project_id,
                )
            if not response:
                self.chat_view.append_message("bot", _("welcome_message"), project_id=project_id)
        except Exception as exc:  # pragma: no cover
            self.chat_view.append_message("Error", str(exc), project_id=project_id)

    def process_action(self, action, params):
        if action == "quick_reply":
            self.chat_view.message_input_view.message_input.setPlainText(params)
            asyncio.ensure_future(self.process_message())
            return True
        return False
