from __future__ import annotations

import json
from typing import Any

from core.I18n import _
from core.ProjectRunner import ProjectRunner
from core.formChat.project_manager import ProjectManager

PREFERRED_CONSOLE_AUTO = "auto"
PREFERRED_CONSOLE_FORCE_VISIBLE = "force_visible"
FOCUS_TOOL_NAMES = {
    "focus_create",
    "focus_update",
    "focus_set_active",
    "focus_list",
    "focus_get_active_summary",
}


class FrontendToolRunner:
    """Handles run_bash_script frontend flow (consent + console display) and focus-related truncation."""

    def __init__(self, chat_view, consent_manager):
        self._chat_view = chat_view
        self._consent_manager = consent_manager
        self._project_manager = ProjectManager.instance()

    async def handle_run_bash_calls(self, tool_calls):
        results = []
        for call in tool_calls:
            fn = call["fn"]
            project_dir = call["project_dir"]
            project_id = call["project_id"]
            args = json.loads(fn.get("arguments", "{}"))
            script_text = args.get("script", "")
            explanation_text = (args.get("explanation") or "").strip()
            preview = script_text.split("\n", 1)[0].strip() or _("(empty script)")

            preferred_strategy = (args.get("preferred_console_strategy") or PREFERRED_CONSOLE_AUTO).lower()
            platform_info = self._normalize_platform_info(args.get("platform_info"))
            forced_console = preferred_strategy == PREFERRED_CONSOLE_FORCE_VISIBLE
            stored_default_show_console = self._project_manager.get_bash_show_console(project_id)
            default_show_console = stored_default_show_console or forced_console

            consent_payload = await self._consent_manager.ensure_bash_consent(
                fn,
                project_id,
                args=args,
                default_show_console=default_show_console,
            )
            final_show_console = consent_payload.get("show_console", False)
            if forced_console and not final_show_console:
                final_show_console = True
                consent_payload["show_console"] = True
                self._project_manager.set_bash_show_console(project_id, True)

            summary = _(
                "run_bash_script (%(console_mode)s): %(explanation)s — %(preview)s"
            ) % {
                "explanation": explanation_text or _("No explanation provided"),
                "preview": preview,
                "console_mode": self._console_mode_label(final_show_console),
            }
            console_meta = _(
                "Platform: %(platform)s | Strategy: %(strategy)s"
            ) % {
                "platform": self._platform_summary(platform_info),
                "strategy": self._describe_console_strategy(preferred_strategy),
            }
            combined_summary = f"{summary}\n{console_meta}"
            self._chat_view.append_message(
                "sys tmp tool",
                combined_summary,
                project_id=project_id,
                apply_markdown=False,
            )

            if consent_payload.get("allowed"):
                self._chat_view.remove_message("sys tmp tool")
                strategy_note = _(" (forced for Git Bash visibility)") if forced_console else ""
                status_message = _(
                    "Executing run_bash_script with %(console_mode)s settings.%(note)s"
                ) % {
                    "console_mode": self._console_mode_label(final_show_console),
                    "note": strategy_note,
                }
                self._chat_view.append_message("sys", status_message, project_id=project_id)
                fn_arguments = json.loads(fn.get("arguments", "{}"))
                fn_arguments["show_console"] = final_show_console
                fn["arguments"] = json.dumps(fn_arguments)
                results.append((project_id, ProjectRunner.run(fn, project_dir)))
            else:
                self._chat_view.remove_message("sys tmp tool")
                self._chat_view.append_message(
                    "sys",
                    _("Execution cancelled for run_bash_script."),
                    project_id=project_id,
                )
        return results

    def _console_mode_label(self, show_console: bool) -> str:
        return _("console visible") if show_console else _("console hidden")

    def _normalize_platform_info(self, info: Any) -> dict[str, Any]:
        if isinstance(info, dict):
            return info
        if isinstance(info, str) and info.strip():
            return {"raw": info.strip()}
        return {}

    def _platform_summary(self, platform_info: dict[str, Any]) -> str:
        if not platform_info:
            return _("unknown")
        parts: list[str] = []
        system = platform_info.get("system")
        if system:
            parts.append(system)
        shell = platform_info.get("shell")
        if shell:
            parts.append(f"shell={shell}")
        arch = platform_info.get("arch")
        if arch:
            parts.append(arch)
        return ", ".join(parts) if parts else _("unknown")

    def _describe_console_strategy(self, strategy: str) -> str:
        mapping = {
            PREFERRED_CONSOLE_FORCE_VISIBLE: _("force visible (Windows Git Bash)"),
            PREFERRED_CONSOLE_AUTO: _("auto"),
        }
        return mapping.get(strategy, strategy or _("auto"))
