from __future__ import annotations

import json
from typing import Any, Dict

from core.I18n import _
from core.formChat.project_manager import ProjectManager
from ui.components.StandardDialog import DialogToolInfo, DialogToolType, StandardDialog

MAX_SCRIPT_PREVIEW = 4000


class BashConsentManager:
    """Handles bash execution consent caching and dialog rendering."""

    def __init__(self, chat_view):
        self._chat_view = chat_view
        self._bash_consent_cache: Dict[str, Dict[str, Any]] = {}

    async def ensure_bash_consent(
        self,
        fn: dict[str, Any],
        project_id: str,
        args: dict[str, Any] | None = None,
        default_show_console: bool = False,
    ) -> dict[str, Any]:
        project_manager = ProjectManager.instance()
        project_dir = project_manager.get_project_path(project_id)
        cache_key = project_id or project_dir or "global"
        cached = self._bash_consent_cache.get(cache_key)
        if cached:
            return cached

        if args is None:
            args = json.loads(fn.get("arguments", "{}"))

        script_text = args.get("script", "") or ""
        explanation_text = (args.get("explanation") or "").strip()
        script_preview = self._build_script_preview(script_text)

        detailed_text = self._build_dialog_body(explanation_text, script_preview)
        dialog_tools = self._build_dialog_tools(default_show_console)

        dialog = StandardDialog.ask_dialog(
            self._chat_view,
            _("Allow CodeGryphon to run this bash script?"),
            detailed_text=detailed_text,
            accept_text=_("Run script"),
            cancel_text=_("Cancel"),
            tools=dialog_tools,
        )

        result = dialog.get_result()
        remember_choice = bool(dialog.get_value_by_tool(dialog_tools[0]))
        show_console_choice = bool(dialog.get_value_by_tool(dialog_tools[1]))

        payload = {"allowed": result, "show_console": show_console_choice}
        project_manager.set_bash_show_console(project_id, show_console_choice)

        if result and remember_choice:
            self._bash_consent_cache[cache_key] = payload
        return payload

    @staticmethod
    def _build_script_preview(script_text: str) -> str:
        if len(script_text) <= MAX_SCRIPT_PREVIEW:
            return script_text
        return f"{script_text[:MAX_SCRIPT_PREVIEW]}\n...[truncated]"

    @staticmethod
    def _build_dialog_body(explanation_text: str, script_preview: str) -> str:
        sections: list[str] = []
        if explanation_text:
            sections.append(_("Explanation:\n%(explanation)s") % {"explanation": explanation_text})
        else:
            sections.append(_("Explanation not provided."))
        sections.append(
            _(
                "Please review the script below. It will be executed within your project directory. \n\n%(script)s"
            )
            % {"script": script_preview}
        )
        return "\n\n".join(sections)

    @staticmethod
    def _build_dialog_tools(default_show_console: bool) -> list[DialogToolInfo]:
        remember_tool = DialogToolInfo(
            id="remember_bash_consent",
            tool_type=DialogToolType.CheckBox,
            title=_("Remember my choice for this session"),
            value=False,
        )
        console_tool = DialogToolInfo(
            id="bash_show_console",
            tool_type=DialogToolType.CheckBox,
            title=_("Show console window during execution"),
            value=default_show_console,
        )
        return [remember_tool, console_tool]
