import asyncio
from typing import Callable

from PyQt6.QtWidgets import QWidget

import api.client as client
from core.I18n import _
from ui.components.StandardDialog import StandardDialog, DialogToolInfo, DialogToolType
from utils.project_info import get_files_hashes, read_file_content


ProgressCallback = Callable[[str], None]


class ProjectSyncService:
    """
    Service for syncing project structure and files.
    Pure logic, no UI except optional dialog for large projects.
    """

    _MAX_PARALLEL_UPLOADS = 4

    def __init__(self, project_manager, parent: QWidget | None = None):
        self.project_manager = project_manager
        self.parent = parent

    async def sync_project_structure(self, project_id, progress_callback: ProgressCallback | None = None, cancel_event=None):
        project_dir = self.project_manager.get_project_path(project_id)
        project = self.project_manager.get_project(project_id)
        project_ext = project.extensions
        project_ignore_dirs = project.ignoreDirs

        if cancel_event and cancel_event.is_set():
            return None

        project_info = self.project_manager.get_project_info(project_id)
        hash_result = get_files_hashes(
            project_dir,
            valid_extensions=project_ext,
            ignore_dirs=project_ignore_dirs,
            file_cache=project_info.sync_cache,
        )
        project_structure = hash_result.hashes
        files_count = len([path for path in project_structure if not path.endswith('/')])
        skip_warning = self.project_manager.get_skip_large_project_warning(project_id, False)
        if files_count > 100 and not skip_warning:
            title = _(
                "This project has more than 100 files. Sending the project structure may take a long time. Do you want to continue?"
            )
            tool = DialogToolInfo(
                id="1",
                tool_type=DialogToolType.CheckBox,
                title=_("Remember my choice for this project"),
            )
            dialog = StandardDialog.ask_dialog(self.parent, title, tools=[tool])
            if not dialog.get_result():
                return None
            if dialog.get_value_by_tool(tool):
                self.project_manager.set_skip_large_project_warning(project_id, True)

        response = await client.send_project_structure(project_structure, project_id)
        while response and len(response) > 0:
            if cancel_event and cancel_event.is_set():
                return None
            await self._upload_missing_files(
                project_dir,
                project_id,
                response,
                project_structure,
                progress_callback,
                cancel_event,
            )
            if cancel_event and cancel_event.is_set():
                return None
            response = await client.send_project_structure(project_structure, project_id)

        project_info.sync_cache = hash_result.cache
        self.project_manager.save_projects_info()
        return project_structure

    async def _upload_missing_files(
        self,
        project_dir: str,
        project_id: str,
        missing_files: dict[str, str],
        project_structure: dict[str, str],
        progress_callback: ProgressCallback | None,
        cancel_event,
    ) -> None:
        file_items = list(missing_files.items())
        semaphore = asyncio.Semaphore(self._MAX_PARALLEL_UPLOADS)
        progress_lock = asyncio.Lock()
        total_files = len(file_items)
        completed = 0

        async def process(file_path: str, file_hash: str):
            nonlocal completed
            if cancel_event and cancel_event.is_set():
                return
            async with semaphore:
                if cancel_event and cancel_event.is_set():
                    return
                full_file_path = file_path.replace("[ROOT]/", project_dir)
                file_content = read_file_content(full_file_path)
                await client.add_file_to_project(file_path, file_content, project_id)
                project_structure[file_path] = file_hash
                if cancel_event and cancel_event.is_set():
                    return
                async with progress_lock:
                    completed += 1
                    if progress_callback:
                        progress_callback(f"({completed}/{total_files}) Analyzing file: {file_path}")

        await asyncio.gather(*(process(path, file_hash) for path, file_hash in file_items))
