import asyncio
import os

from PyQt6.QtCore import pyqtSignal, QObject
from PyQt6.QtWidgets import QWidget

import api.client as client
from core.I18n import _
from core.formChat.project_sync_service import ProjectSyncService
from models.ProjectConfigModel import ProjectsConfigModel, ProjectConfigModel
from models.ProjectModel import ProjectModel
from services.spellcheck.SpellcheckService import SpellcheckService
from ui.components.StandardDialog import StandardDialog


def normalize_project_path(path: str) -> str:
    """Normalize the project path to ensure it ends with an OS-specific separator."""
    if path and not path.endswith(os.sep):
        return path + os.sep
    return path


class ProjectManager(QObject):
    projectsUpdated = pyqtSignal(list)
    selectedProjectChanged = pyqtSignal(object)
    beforeProjectChanged = pyqtSignal(str)
    appendMessage = pyqtSignal(str)

    __instance = None

    def __init__(self, parent: QWidget | None = None):
        super().__init__(parent)
        self.parent = parent
        self.sync_service = ProjectSyncService(self, parent)
        self._unread_projects: set[str] = set()
        self._projects: list[ProjectModel] = []
        self._selected_project: ProjectModel | None = None
        self._projects_info: ProjectsConfigModel = ProjectsConfigModel.load()
        self._current_project: ProjectConfigModel | None = None
        self._spellcheck_service = SpellcheckService(projects_config=self._projects_info)
        ProjectManager.__instance = self

    @staticmethod
    def instance(parent: QWidget | None = None):
        if not ProjectManager.__instance:
            ProjectManager.__instance = ProjectManager(parent=parent)
        return ProjectManager.__instance

    @staticmethod
    def clear():
        ProjectManager.__instance = None

    @property
    def project_id(self) -> str | None:
        return self._selected_project.id if self._selected_project else None

    @property
    def projects(self) -> list[ProjectModel]:
        return self._projects

    @property
    def selectedProject(self) -> ProjectModel | None:
        return self._selected_project

    @property
    def spellcheck_service(self) -> SpellcheckService:
        return self._spellcheck_service

    def get_project(self, project_id: str) -> ProjectModel | None:
        for project in self._projects:
            if project.id == project_id:
                return project
        return None

    def __selectProject(self, item: ProjectModel | None):
        self._selected_project = item
        self.selectedProjectChanged.emit(item)

    def __set_projects(self, projects: list[ProjectModel], notify: bool = True):
        self._projects = projects
        if notify:
            self.projectsUpdated.emit(projects)

    def load_projects_info(self):
        self._projects_info = ProjectsConfigModel.load()
        self._spellcheck_service.reload(self._projects_info)

    def save_projects_info(self):
        self._projects_info.save()

    def get_project_info(self, project_id: str) -> ProjectConfigModel:
        model = self._projects_info.projects.get(project_id)
        if model:
            return model
        default_model = ProjectConfigModel(
            id=project_id,
            path=None,
            skip_large_project_warning=None,
            bash_show_console=True,
        )
        self._projects_info.projects[project_id] = default_model
        return default_model

    def set_project_info_path(self, project_id: str, project_path: str) -> None:
        project_path = normalize_project_path(project_path)
        project_info = self.get_project_info(project_id)
        project_info.path = project_path
        self.save_projects_info()

    def set_bash_show_console(self, project_id: str, show_console: bool) -> None:
        project_info = self.get_project_info(project_id)
        project_info.bash_show_console = show_console
        self.save_projects_info()

    def get_bash_show_console(self, project_id: str, default: bool = True) -> bool:
        project_info = self.get_project_info(project_id)
        if project_info.bash_show_console is not None:
            return project_info.bash_show_console
        return default

    def get_project_path(self, project_id: str) -> str | None:
        project_info = self.get_project_info(project_id)
        return project_info.path

    def set_skip_large_project_warning(self, project_id: str, flag_value: bool):
        project_info = self.get_project_info(project_id)
        project_info.skip_large_project_warning = flag_value
        self.save_projects_info()

    def get_skip_large_project_warning(self, project_id: str, default: bool = False) -> bool:
        project_info = self.get_project_info(project_id)
        if project_info.skip_large_project_warning is None:
            return default
        return project_info.skip_large_project_warning

    def set_default_project(self, project_id: str):
        self._projects_info.default = project_id
        self.save_projects_info()

    async def load_default_project(self):
        project_id = self._projects_info.default
        if project_id:
            await self._select_project_async(project_id)

    async def update_projects(self, notify: bool = True):
        projects = await client.project_list()
        self.__set_projects(projects, notify=notify)

    async def load_projects_list(self):
        try:
            selected_project_id = self._projects_info.default
            self._projects.clear()
            await self.update_projects()

            if selected_project_id:
                found = False
                for item in self._projects:
                    if item.id == selected_project_id:
                        self.__selectProject(item)
                        found = True
                        break
                if not found and self._projects:
                    first_item = self._projects[0]
                    self.__selectProject(first_item)
                    await self._select_project_async(first_item.id)
                elif not found and not self._projects:
                    self.__selectProject(None)
                    self._current_project = None
            else:
                if self._projects:
                    first_item = self._projects[0]
                    self.__selectProject(first_item)
                    await self._select_project_async(first_item.id)
                else:
                    self.__selectProject(None)
                    self._current_project = None

        except Exception as exc:
            msg = _("Failed to load projects list")
            self.appendMessage.emit(f"{msg}: {exc}")

    def update_project_list_notifications(self):
        asyncio.ensure_future(self.load_projects_list())

    async def _select_project_async(self, project_id: str):
        prev_id = self._current_project.id if self._current_project else None
        if prev_id == project_id:
            return
        if prev_id:
            self.beforeProjectChanged.emit(prev_id)
        try:
            project_info = await client.project_get(project_id)
        except Exception:
            self.__selectProject(None)
            self._current_project = None
            return
        project_dir = self.get_project_path(project_id)
        bash_show_console = self.get_bash_show_console(project_id)

        self._current_project = ProjectConfigModel(
            id=project_id,
            path=project_dir,
            skip_large_project_warning=None,
            bash_show_console=bash_show_console,
        )

        for item in self._projects:
            if item.id == project_id:
                self.__selectProject(item)
                break
        self.set_default_project(project_id)

    async def _project_archive(self, project_id: str):
        reply = StandardDialog.ask_remove(
            self.parent,
            _("Are you sure you want to archive this project?"),
            accept_text=_("Archive"),
        )
        if reply:
            try:
                await client.project_update(project_id, archived=True)
                await self.load_projects_list()
            except Exception as exc:
                print(f"Failed to archive project: {exc}")

    async def sync_project_structure(self, progress_callback=None, project_id=None, cancel_event=None):
        if project_id is None:
            project_id = self._current_project.id if self._current_project else None
        return await self.sync_service.sync_project_structure(
            project_id,
            progress_callback=progress_callback,
            cancel_event=cancel_event,
        )

    def mark_project_unread(self, project_id: str):
        if project_id not in self._unread_projects:
            self._unread_projects.add(project_id)
            self.update_project_list_notifications()

    def clear_project_unread(self, project_id: str):
        if project_id in self._unread_projects:
            self._unread_projects.remove(project_id)
            self.update_project_list_notifications()

    def has_unread(self, project_id: str) -> bool:
        return project_id in self._unread_projects

    def update_project_info(self, project_id: str, project_dir: str):
        asyncio.ensure_future(self.__update_project_info(project_id, project_dir))

    async def __update_project_info(self, project_id: str, project_dir: str):
        await self.load_projects_list()
        self.set_project_info_path(project_id, project_dir)
        await self._select_project_async(project_id)

    async def add_extensions(self, extensions: list[str], project_id: str | None = None):
        project = None
        if project_id is None:
            project = self.selectedProject
            project_id = project.id if project else None
        else:
            for p in self.projects:
                if p.id == project_id:
                    project = p
                    break
        if project is None or project_id is None:
            return
        extensions = list(dict.fromkeys(project.extensions + extensions))
        if set(project.extensions) == set(extensions):
            return
        valid_extensions = ",".join(extensions)
        await client.project_update(project_id, valid_extensions=valid_extensions)
        await self.update_projects(notify=False)
