import asyncio

from PyQt6.QtWidgets import QMessageBox

from api import client
from ui.formChat.FormProjectInfo import UIFormProjectInfo
from core.I18n import _


class FormProjectInfo(UIFormProjectInfo):
    def __init__(self, parent, project_id=None):
        super().__init__(parent, project_id)

        self.project_id = project_id

        if project_id is not None:
            asyncio.ensure_future(self.load_project_details(project_id))

    async def load_project_details(self, project_id):
        try:
            project = await client.project_get(project_id)
            self.project_name_input.setText(project['name'])
            self.project_description_input.setText(project['description'])
            self.project_valid_extensions_input.setText(project['valid_extensions'])
            self.ignore_dirs_input.setText(project['ignore_dirs'])
            self.project_dir_input.setText(self.parent().projects_view.project_manager.get_project_path(project_id))
            self.project_rules_input.setText(project.get('rules', ''))
        except Exception as e:
            QMessageBox.warning(self, _("Error"), f"{_('Failed to load project details')}: {e}")

    def on_create_project_button_clicked(self):
        project_name = self.project_name_input.text().strip()
        project_dir = self.project_dir_input.text().strip()
        project_description = self.project_description_input.toPlainText().strip()
        project_valid_extensions = self.project_valid_extensions_input.text().strip().replace(' ', '')
        project_ignore_dirs = self.ignore_dirs_input.text().strip().replace(' ', '')
        project_rules = self.project_rules_input.toPlainText().strip()
        if not project_name:
            QMessageBox.warning(self, _("Validation"), _("Project Name is required"))
            return
        if not project_dir:
            QMessageBox.warning(self, _("Validation"), _("Project Directory is required"))
            return
        if not project_valid_extensions:
            QMessageBox.warning(self, _("Validation"), _("Project Valid Extensions is required"))
            return
        if self.project_id is None:
            asyncio.ensure_future(
                self.create_project(project_name, project_description, project_valid_extensions, project_ignore_dirs, project_dir, project_rules))
        else:
            asyncio.ensure_future(
                self.update_project(self.project_id, project_name, project_description, project_valid_extensions, project_ignore_dirs, project_dir, project_rules))

    async def create_project(self, project_name, project_description, project_valid_extensions, project_ignore_dirs, project_dir, project_rules):
        try:
            response = await client.project_create(project_name, project_description, project_valid_extensions, project_ignore_dirs, project_rules)
            if 'id' in response:
                project_id = response['id']
                self.parent().projects_view.project_manager.update_project_info(project_id, project_dir)
                self.close()
        except Exception as e:
            QMessageBox.warning(self, _("Error"), f"{_('Failed to create project')}: {e}")

    async def update_project(self, project_id, project_name, project_description, project_valid_extensions, project_ignore_dirs, project_dir, project_rules):
        try:
            await client.project_update(project_id, project_name, project_description, project_valid_extensions, project_ignore_dirs, project_rules)
            self.parent().projects_view.project_manager.update_project_info(project_id, project_dir)
            self.close()
        except Exception as e:
            QMessageBox.warning(self, _("Error"), f"{_('Failed to update project')}: {e}")
