import os

from PyQt6.QtCore import Qt

from core.FormAppSettings import FormAppSettings
from models.ProjectModel import ProjectModel
from ui.formChat.FormChat import UIFormChat


def normalize_project_path(path: str) -> str:
    """
    Normalize the project path to ensure it ends with an OS-specific separator.
    """
    if path and not path.endswith(os.sep):
        return path + os.sep
    return path

class FormChat(UIFormChat):
    def __init__(self):
        super().__init__()
        # Sound settings
        self._always_on_top = True
        self._opacity = 1.0
        # --- Store input drafts per project ---
        self._drafts = {}

        # Connect UI signals
        self.settings_button.clicked.disconnect()
        self.settings_button.clicked.connect(self.open_settings)

        # --- FOR LIVE OPACITY UPDATE ---
        self._last_always_on_top = self._always_on_top

        self.apply_window_properties()
        self._binds()

    def _binds(self):
        self.projects_view.project_manager.selectedProjectChanged.connect(self.__selectedProjectChanged)
        self.projects_view.project_manager.beforeProjectChanged.connect(self.__beforeProjectChanged)
        self.projects_view.project_manager.appendMessage.connect(lambda message: self.chatView.append_message('Error', message))

    def __selectedProjectChanged(self, projectItem: ProjectModel | None):
        self.chatView.setEnabled(projectItem is not None)
        self.chatView.message_input.clear()
        if projectItem is not None:
            self.chatView.load_chat_history() # self.message_handler.load_history()
            self.chatView.show_pending_messages(projectItem.id)

            draft_text = self._drafts.get(projectItem.id, "")
            self.chatView.message_input.setPlainText(draft_text)
            cursor = self.chatView.message_input.textCursor()
            from PyQt6.QtGui import QTextCursor
            cursor.movePosition(QTextCursor.MoveOperation.End)
            self.chatView.message_input.setTextCursor(cursor)
            self.chatView.message_input.setFocus()

    def __beforeProjectChanged(self, prev_id: str):
        self._drafts[prev_id] = self.chatView.message_input.toPlainText()

    def apply_window_properties(self, force_all=False):
        always_on_top = self.settings_manager.app_settings.always_on_top
        opacity = self.settings_manager.app_settings.opacity
        # Apply always-on-top ONLY when needed, opacity ALWAYS (live update)
        # If force_all=True: apply both, otherwise only changed flags
        flags = self.windowFlags() & ~Qt.WindowType.WindowStaysOnTopHint
        if force_all or always_on_top != self._last_always_on_top:
            if always_on_top:
                flags = flags | Qt.WindowType.WindowStaysOnTopHint
            self.setWindowFlags(flags)
            self._last_always_on_top = always_on_top
            self.show()  # Only needed after setWindowFlags
        self.setWindowOpacity(max(0.4, min(1.0, self._opacity)))

    def __getattr__(self, name):
        for manager in (self.projects_view.project_manager, self.message_handler, self.autocomplete_manager, self.settings_manager):
            if hasattr(manager, name):
                attr = getattr(manager, name)
                if callable(attr):
                    def method(*args, **kwargs):
                        return attr(*args, **kwargs)
                    return method
                else:
                    return attr
        raise AttributeError(f"'{self.__class__.__name__}' object has no attribute '{name}'")

    def open_settings(self):
        settings_form = FormAppSettings(self)
        # Live update opacity instantly when slider moves
        settings_form.slider_opacity.valueChanged.connect(lambda _: self.on_window_opacity_slider_changed(settings_form))
        # Live update always-on-top also triggers re-flagging (with window recreation)
        settings_form.checkbox_always_on_top.stateChanged.connect(lambda _: self.on_window_always_on_top_changed(settings_form))
        settings_form.setWindowModality(Qt.WindowModality.ApplicationModal)
        settings_form.exec()
        # On dialog close, save changes merging all previous data
        self.apply_settings_from_form_and_save(settings_form)
        self.apply_window_properties(force_all=True)

    def on_window_opacity_slider_changed(self, settings_form):
        self._opacity = max(0.4, settings_form.slider_opacity.value() / 100.0)
        self.setWindowOpacity(self._opacity)
        # print current opacity for debugging
        self.repaint()

    def on_window_always_on_top_changed(self, settings_form):
        self._always_on_top = settings_form.checkbox_always_on_top.isChecked()
        self.apply_window_properties(force_all=True)

    def apply_settings_from_form_and_save(self, settings_form):
        self.settings_manager.app_settings.enable_sound  = settings_form.checkbox_enable_sound.isChecked()
        from core.SoundManager import SoundManager
        self.settings_manager.app_settings.volume = int(max(0, min(100, settings_form.slider_volume.value())))
        SoundManager.instance().set_volume(self.settings_manager.app_settings.sound_volume, with_save=False)
        self.settings_manager.app_settings.enable_autocomplete= settings_form.checkbox_enable_autocomplete.isChecked()
        self.settings_manager.app_settings.always_on_top = settings_form.checkbox_always_on_top.isChecked()
        self.settings_manager.app_settings.opacity = int(max(40, min(100, settings_form.slider_opacity.value())))
        SoundManager.instance().set_is_enabled(self.settings_manager.app_settings.enable_sound)
        self.chatView.autocomplete_manager.set_enabled(self.settings_manager.app_settings.enable_autocomplete)
        self._always_on_top = self.settings_manager.app_settings.always_on_top
        self._opacity = self.settings_manager.app_settings.opacity / 100.0
        self.settings_manager.save_settings()
