from __future__ import annotations

import asyncio
from dataclasses import asdict, dataclass, field
from typing import Any, Dict, List, Optional, Set

from PyQt6.QtCore import QObject, QTimer, pyqtSignal

import api.client as client


@dataclass(slots=True)
class FocusSubtask:
    id: str | None = None
    text: str = ""
    status: str = "WAITING"
    is_done: bool = False
    created_at: str | None = None


@dataclass(slots=True)
class FocusPlanStep:
    id: str | None = None
    text: str = ""
    status: str = "WAITING"
    is_done: bool = False
    subtasks: List[FocusSubtask] = field(default_factory=list)


@dataclass(slots=True)
class FocusActiveSummary:
    id: str | None = None
    project_id: str | None = None
    created_by_user_id: str | None = None
    assigned_user_id: str | None = None
    assignee_label: str | None = None
    title: str | None = None
    description: str | None = None
    status: str = "DRAFT"
    plan_text: str | None = None
    plan_preview: str | None = None
    plan_steps: List[FocusPlanStep] | None = None
    plan_structure: dict | None = None
    active_step_subtasks: List[FocusSubtask] | None = None
    completed_steps: int = 0
    total_steps: int = 0
    progress_percent: int = 0
    updated_at: str | None = None
    priority: int = 3
    total_time_seconds: int = 0
    glyphs_spent: int = 0
    messages_count: int = 0
    changed_files_count: int = 0
    new_rows_count: int = 0

    def __post_init__(self):
        if self.plan_steps is None:
            self.plan_steps = []
        if self.active_step_subtasks is None:
            self.active_step_subtasks = []
        if self.plan_structure is None:
            self.plan_structure = {"version": 1, "steps": []}


@dataclass(slots=True)
class FocusEntry:
    id: str
    project_id: str
    title: str
    plan_text: str
    plan_structure: dict | None
    is_active: bool
    created_by_user_id: str
    assigned_user_id: str | None
    description: str | None = None
    status: str = "DRAFT"
    assigned_user_label: str | None = None
    created_at: str | None = None
    updated_at: str | None = None
    priority: int = 3

    @property
    def progress(self) -> tuple[int, int]:
        steps = (self.plan_structure or {}).get("steps") or []
        if steps:
            done = sum(1 for step in steps if step.get("is_done"))
            return done, len(steps)
        lines = [line.strip() for line in self.plan_text.splitlines() if line.strip()]
        done = sum(1 for line in lines if line.lower().startswith("[done]"))
        return done, len(lines)

    @property
    def assignee_display(self) -> str:
        if self.assigned_user_label:
            return self.assigned_user_label
        if self.assigned_user_id:
            return f"…{self.assigned_user_id[-6:]}"
        return "—"


class FocusManager(QObject):
    focus_list_updated = pyqtSignal(str, list)
    focus_active_updated = pyqtSignal(str, object)

    AUTO_REFRESH_INTERVAL_MS = 60_000

    _instance: "FocusManager" | None = None

    def __init__(self):
        super().__init__()
        self._focuses: Dict[str, list[FocusEntry]] = {}
        self._active_summary_cache: Dict[str, dict[str, Any] | None] = {}
        self._lock = asyncio.Lock()
        self._refresh_timer = QTimer(self)
        self._refresh_timer.setSingleShot(True)
        self._refresh_timer.setInterval(self.AUTO_REFRESH_INTERVAL_MS)
        self._refresh_timer.timeout.connect(self._handle_timer_tick)
        self._active_project_id: str | None = None
        self._timer_refresh_inflight = False
        self._pending_manual_refreshes: Set[str] = set()

    @classmethod
    def instance(cls) -> "FocusManager":
        if cls._instance is None:
            cls._instance = FocusManager()
        return cls._instance

    def set_active_project(self, project_id: str | None) -> None:
        if project_id == self._active_project_id:
            if project_id:
                self._restart_timer()
            return
        self._active_project_id = project_id
        if project_id:
            self._restart_timer()
        else:
            self._refresh_timer.stop()

    def clear_active_project(self) -> None:
        self.set_active_project(None)

    def pause_auto_refresh(self, project_id: str | None = None) -> None:
        if project_id is None or project_id == self._active_project_id:
            self._refresh_timer.stop()

    def resume_auto_refresh(self, project_id: str | None = None) -> None:
        target = project_id or self._active_project_id
        if target and target == self._active_project_id:
            self._restart_timer()

    def bump_auto_refresh_timer(self) -> None:
        if self._refresh_timer.isActive():
            self._refresh_timer.start()

    def trigger_immediate_refresh(self, project_id: str | None) -> None:
        if not project_id or project_id != self._active_project_id:
            return
        if project_id in self._pending_manual_refreshes:
            return
        self._pending_manual_refreshes.add(project_id)
        asyncio.ensure_future(self._run_manual_refresh(project_id))

    async def refresh_focuses(self, project_id: str) -> list[FocusEntry]:
        async with self._lock:
            raw = await client.focus_list(project_id)
            items = [self._parse_focus(item) for item in raw.get("focuses", [])]
            self._focuses[project_id] = items
        self.focus_list_updated.emit(project_id, self.get_focuses(project_id))
        self._touch_auto_refresh_timer(project_id)
        return self.get_focuses(project_id)

    async def refresh_active_summary(self, project_id: str) -> dict[str, Any] | None:
        summary = await client.focus_active_summary(project_id)
        normalized_summary = self._normalize_active_summary(summary)
        self._active_summary_cache[project_id] = normalized_summary
        self.focus_active_updated.emit(project_id, normalized_summary)
        self._touch_auto_refresh_timer(project_id)
        return normalized_summary

    async def refresh_all(self, project_id: str) -> None:
        await asyncio.gather(self.refresh_focuses(project_id), self.refresh_active_summary(project_id))

    def get_focuses(self, project_id: str) -> list[FocusEntry]:
        return list(self._focuses.get(project_id, []))

    def get_active_summary(self, project_id: str) -> dict[str, Any] | None:
        return self._active_summary_cache.get(project_id)

    async def warm_cache(self, project_id: str) -> None:
        await self.refresh_all(project_id)

    async def create_focus(
        self,
        project_id: str,
        *,
        title: str,
        plan_text: str,
        description: str | None = None,
        status: str | None = None,
        set_active: bool = True,
        assigned_user_id: str | None = None,
        priority: int = 3,
        plan_structure: dict | None = None,
    ) -> dict[str, Any]:
        resp = await client.focus_create(
            project_id,
            title=title,
            plan_text=plan_text,
            description=description,
            status=status,
            set_active=set_active,
            assigned_user_id=assigned_user_id,
            priority=priority,
            plan_structure=plan_structure,
        )
        await self.refresh_all(project_id)
        return resp

    async def update_focus(
        self,
        focus_id: str,
        *,
        title: str | None = None,
        plan_text: str | None = None,
        description: str | None = None,
        status: str | None = None,
        assigned_user_id: str | None = None,
        priority: int | None = None,
        plan_structure: dict | None = None,
    ) -> dict[str, Any]:
        resp = await client.focus_update(
            focus_id,
            title=title,
            plan_text=plan_text,
            description=description,
            status=status,
            assigned_user_id=assigned_user_id,
            priority=priority,
            plan_structure=plan_structure,
        )
        project_id = self._resolve_project_id(focus_id, resp)
        if project_id:
            await self.refresh_all(project_id)
        return resp

    async def set_active(self, focus_id: str) -> dict[str, Any]:
        resp = await client.focus_set_active(focus_id)
        project_id = self._resolve_project_id(focus_id, resp)
        if project_id:
            await self.refresh_all(project_id)
        return resp

    def _find_focus_by_id(self, focus_id: str) -> FocusEntry | None:
        for focuses in self._focuses.values():
            for focus in focuses:
                if focus.id == focus_id:
                    return focus
        return None

    def _resolve_project_id(self, focus_id: str, response: dict[str, Any]) -> str | None:
        focus = self._find_focus_by_id(focus_id)
        if focus:
            return focus.project_id
        return response.get("project_id")

    def _restart_timer(self) -> None:
        self._refresh_timer.stop()
        if self._active_project_id:
            self._refresh_timer.start()

    def _touch_auto_refresh_timer(self, project_id: str) -> None:
        if project_id == self._active_project_id:
            self._restart_timer()

    def _handle_timer_tick(self) -> None:
        project_id = self._active_project_id
        if not project_id or self._timer_refresh_inflight:
            self._schedule_next_tick()
            return
        self._timer_refresh_inflight = True
        asyncio.ensure_future(self._run_timer_refresh(project_id))

    def _schedule_next_tick(self) -> None:
        if self._active_project_id:
            self._refresh_timer.start()

    async def _run_timer_refresh(self, project_id: str) -> None:
        try:
            await self.refresh_all(project_id)
        finally:
            self._timer_refresh_inflight = False
            self._schedule_next_tick()

    async def _run_manual_refresh(self, project_id: str) -> None:
        try:
            await self.refresh_all(project_id)
        finally:
            self._pending_manual_refreshes.discard(project_id)
            self._touch_auto_refresh_timer(project_id)

    @staticmethod
    def _parse_focus(item: Dict[str, Any]) -> FocusEntry:
        return FocusEntry(
            id=item.get("id", ""),
            project_id=item.get("project_id", ""),
            title=item.get("title", ""),
            plan_text=item.get("plan_text", ""),
            plan_structure=item.get("plan_structure"),
            is_active=item.get("is_active", False),
            created_by_user_id=item.get("created_by_user_id", ""),
            assigned_user_id=item.get("assigned_user_id"),
            description=item.get("description"),
            status=item.get("status", "DRAFT"),
            assigned_user_label=item.get("assigned_user_label") or item.get("assignee_label"),
            created_at=item.get("created_at"),
            updated_at=item.get("updated_at"),
            priority=int(item.get("priority", 3)) if item.get("priority") is not None else 3,
        )

    @staticmethod
    def _normalize_active_summary(summary: Optional[Dict[str, Any]]) -> Dict[str, Any] | None:
        if not summary:
            return None
        steps_data = summary.get("plan_steps") or []
        plan_steps = [
            FocusPlanStep(
                id=step.get("id"),
                text=step.get("text", ""),
                status=step.get("status", "TODO"),
                is_done=bool(step.get("is_done", False)),
                subtasks=[
                    FocusSubtask(
                        id=subtask.get("id"),
                        text=subtask.get("text", ""),
                        status=subtask.get("status", "WAITING"),
                        is_done=bool(subtask.get("is_done", False)),
                        created_at=subtask.get("created_at"),
                    )
                    for subtask in (step.get("subtasks") or [])
                ],
            )
            for step in steps_data
        ]
        active_subtasks = [
            FocusSubtask(
                id=subtask.get("id"),
                text=subtask.get("text", ""),
                status=subtask.get("status", "WAITING"),
                is_done=bool(subtask.get("is_done", False)),
                created_at=subtask.get("created_at"),
            )
            for subtask in summary.get("active_step_subtasks", [])
        ]
        normalized = FocusActiveSummary(
            id=summary.get("id"),
            project_id=summary.get("project_id"),
            created_by_user_id=summary.get("created_by_user_id"),
            assigned_user_id=summary.get("assigned_user_id"),
            assignee_label=summary.get("assignee_label") or summary.get("assigned_user_label"),
            title=summary.get("title"),
            description=summary.get("description"),
            status=summary.get("status", "DRAFT"),
            plan_text=summary.get("plan_text"),
            plan_preview=summary.get("plan_preview"),
            plan_steps=plan_steps,
            plan_structure=summary.get("plan_structure"),
            active_step_subtasks=active_subtasks,
            completed_steps=summary.get("completed_steps", 0),
            total_steps=summary.get("total_steps", 0),
            progress_percent=summary.get("progress_percent", 0),
            updated_at=summary.get("updated_at"),
            priority=int(summary.get("priority", 3)) if summary.get("priority") is not None else 3,
            total_time_seconds=int(summary.get("total_time_seconds", 0) or 0),
            glyphs_spent=int(summary.get("glyphs_spent", 0) or 0),
            messages_count=int(summary.get("messages_count", 0) or 0),
            changed_files_count=int(summary.get("changed_files_count", 0) or 0),
            new_rows_count=int(summary.get("new_rows_count", 0) or 0),
        )
        return {
            "id": normalized.id,
            "project_id": normalized.project_id,
            "created_by_user_id": normalized.created_by_user_id,
            "assigned_user_id": normalized.assigned_user_id,
            "assignee_label": normalized.assignee_label,
            "title": normalized.title,
            "description": normalized.description,
            "status": normalized.status,
            "plan_text": normalized.plan_text,
            "plan_preview": normalized.plan_preview,
            "plan_steps": [asdict(step) for step in normalized.plan_steps],
            "plan_structure": normalized.plan_structure,
            "active_step_subtasks": [asdict(subtask) for subtask in normalized.active_step_subtasks],
            "completed_steps": normalized.completed_steps,
            "total_steps": normalized.total_steps,
            "progress_percent": normalized.progress_percent,
            "updated_at": normalized.updated_at,
            "priority": normalized.priority,
            "total_time_seconds": normalized.total_time_seconds,
            "glyphs_spent": normalized.glyphs_spent,
            "messages_count": normalized.messages_count,
            "changed_files_count": normalized.changed_files_count,
            "new_rows_count": normalized.new_rows_count,
        }
