from typing import Optional, Callable, Any, TypeVar, ClassVar

from pydantic import BaseModel

from core.config.ConfigLoader import ConfigLoader

T = TypeVar("T", bound="ConfigModel")

class ConfigModel(BaseModel):
    """
    Базовый класс для конфиг-моделей.
    Добавляет методы .load() / .save() / .path() / .set_path() / .loader().
    Работает через ваш ConfigLoader.
    """

    __config_path__: ClassVar[str] = "tmp.json"
    __migrate_func__: ClassVar[Optional[Callable[[Any], Any]]] = None
    __config_loader__: ClassVar[Optional["ConfigLoader"]] = None
    version: str | None = "1"

    # ---- API ----

    @classmethod
    def path(cls) -> str:
        return cls.__config_path__

    @classmethod
    def set_path(cls, new_path: str) -> None:
        cls.__config_path__ = new_path
        cls.__config_loader__ = None  # сброс кэша

    @classmethod
    def loader(cls: type[T]) -> "ConfigLoader[T]":
        if cls.__config_loader__ is None:
            cls.__config_loader__ = ConfigLoader(
                path=cls.path(),
                model=cls,
                migrateFunc=cls.__migrate_func__,
            )
        return cls.__config_loader__

    @classmethod
    def load(cls: type[T]) -> T:
        return cls.loader().load()

    def save(
        self: T,
        *,
        indent: int = 2,
        by_alias: bool = False,
        exclude_none: bool = True,
    ) -> None:
        type(self).loader().save(
            self,
            indent=indent,
            by_alias=by_alias,
            exclude_none=exclude_none,
        )
