import json
from typing import Generic, TypeVar, Type, Optional, Callable, Any

from pydantic import BaseModel, TypeAdapter, ValidationError

from utils import universal_config_path, log_info

T = TypeVar("T", bound=BaseModel)

class ConfigLoader(Generic[T]):
    def __init__(self, path: str, model: Type[T], migrateFunc: Callable[[Any], T] = None):
        self._path = universal_config_path(path)
        self._model = model
        self._adapter: TypeAdapter[T] = TypeAdapter(model)
        self._migrateFunc: Callable[[Any], T] = migrateFunc

    def load(self) -> Optional[T]:
        log_info(f"[ConfigLoader] Start load from {self._path}")
        try:
            with open(self._path, 'r', encoding='utf-8') as f:
                raw = f.read()
        except FileNotFoundError:
            log_info(f"[ConfigLoader] File not found")
            return self.__get_default()
        if not raw:
            log_info(f"[ConfigLoader] file is empty")
            return self.__get_default()
        try:
            # 3) если есть миграция — применим
            if self._migrateFunc:
                migrated = self.__migrate()
                if migrated:
                    return migrated
            return self._adapter.validate_json(raw)
        except ValidationError:
            log_info(f"[ConfigLoader] try soft load")
            # 4) необязательный «мягкий» режим: доливаем дефолты и валидируем ещё раз
            try:
                data = json.loads(raw) or {}
                if not isinstance(data, dict):
                    return self._model()
                defaults = self._model().model_dump()
                merged = {**defaults, **data}  # неглубокое слияние
                return self._model.model_validate(merged)
            except Exception:
                log_info(f"[ConfigLoader] Return default")
                return self.__get_default()

    def __migrate(self) -> Optional[T]:
        if not self._migrateFunc:
            return None
        try:
            with open(self._path, 'r', encoding='utf-8') as f:
                json_str = json.load(f)
                result = self._migrateFunc(json_str)
                self.save(result)
                log_info(f"[ConfigLoader] Successfully load from {self._path}")
                return result
        except Exception:
            return None

    def save(self, obj: T, indent: int = 2, by_alias: bool = False, exclude_none = True) -> None:
        """
                Сохраняем экземпляр модели T в JSON.
                - indent: для «красивой» печати
                - by_alias: сериализовать по алиасам (если задавались Field(alias=...))
                """
        # Создаём директорию, если нужно
        # import os
        # os.makedirs(os.path.dirname(self._path) or ".", exist_ok=True)

        try:
            json_str = obj.model_dump_json(indent=indent, by_alias=by_alias, exclude_none=exclude_none)
            with open(self._path, 'w', encoding='utf-8') as f:
                f.write(json_str)
            log_info(f"[ConfigLoader] Successfully saved to {self._path}")
        except Exception as e:
            print(f"Failed to save {self._path}: {e}")

    def __get_default(self) -> T:
        model = self._model()
        self.save(model)
        return model
