from threading import RLock
from typing import Optional, TypeVar, ClassVar, Type

from core.config.ConfigModel import ConfigModel

T = TypeVar("T", bound="CachedConfigModel")

class CachedConfigModel(ConfigModel):
    """
    Надстройка над ConfigMixin:
    - load(): возвращает закешированный экземпляр, если он уже был загружен;
              иначе грузит с диска, кеширует и возвращает.
    - reload(): принудительно перечитывает с диска и обновляет кеш.
    - set_path(): сбрасывает кеш (чтобы новый путь не использовал старые данные).
    """

    __cached_instance__: ClassVar[Optional["CachedConfigMixin"]] = None
    __cache_lock__: ClassVar[RLock] = RLock()

    @classmethod
    def load(cls: Type[T]) -> T:
        with cls.__cache_lock__:
            if isinstance(cls.__cached_instance__, cls):
                return cls.__cached_instance__  # type: ignore[return-value]

            inst: T= super().load()  # загрузка через ConfigMixin/ConfigLoader
            cls.__cached_instance__ = inst
            return inst

    @classmethod
    def reload(cls: Type[T]) -> T:
        """Принудительная перезагрузка с диска (игнорируя кеш)."""
        with cls.__cache_lock__:
            inst: T = super().load()
            cls.__cached_instance__ = inst
            return inst

    @classmethod
    def set_path(cls, new_path: str) -> None:
        """Меняем путь и сбрасываем кеш, чтобы следующая load() читала новый файл."""
        with cls.__cache_lock__:
            cls.__cached_instance__ = None
            super().set_path(new_path)

    def save(self, * , indent: int = 2, by_alias: bool = False, exclude_none: bool = True) -> None:
        """
        Сохраняем текущий инстанс и оставляем его в кеше.
        (Если хотите, можно сделать атомарную запись внутри ConfigLoader.save.)
        """
        type(self).loader().save(self, indent=indent, by_alias=by_alias, exclude_none=exclude_none)
        # гарантируем, что кеш указывает на актуальный объект
        type(self).__cached_instance__ = self