import asyncio

import api.client as client
from core.settings_manager import SettingsManager


class AutocompleteManager:
    def __init__(self, chat_view):
        self.chat_view = chat_view
        self.autocomplete_task = None
        self.autocomplete_timer = None  # Will be set externally
        self.__is_enabled = SettingsManager.instance().app_settings.enable_autocomplete

    @property
    def is_enabled(self):
        return self.__is_enabled

    def set_enabled(self, enabled: bool):
        self.__is_enabled = enabled
        SettingsManager.instance().app_settings.enable_autocomplete = enabled
        SettingsManager.instance().save_settings()

    def set_timer(self, timer):
        self.autocomplete_timer = timer

    def _start_timer(self):
        if self.autocomplete_timer and self.__is_enabled:
            self.autocomplete_timer.start()

    def on_message_input_changed(self):
        # Clear inline autocomplete suggestion immediately on user input
        self.chat_view.message_input_view.message_input.clear_suggestion()

        # Reset debounce timer
        if self.autocomplete_timer:
            self.autocomplete_timer.stop()

        # Start timer only if autocomplete is enabled
        self._start_timer()

        # Cancel ongoing autocomplete task
        if self.autocomplete_task and not self.autocomplete_task.done():
            self.autocomplete_task.cancel()

    def on_autocomplete_timeout(self):
        if not self.__is_enabled:
            self.chat_view.hide_autocomplete_inline()
            return

        text = self.chat_view.message_input_view.message_input.toPlainText().strip()
        if not text:
            self.chat_view.hide_autocomplete_inline()
            return

        self.autocomplete_task = asyncio.ensure_future(self._autocomplete(text))

    async def _autocomplete(self, text):
        # Send partial text and last assistant message for context
        try:
            suggestion = await client.autocomplete_message(text, self.chat_view.last_received_message)
            if suggestion:
                if suggestion.startswith(text):
                    suggestion = suggestion[len(text):]
                self.chat_view.show_autocomplete_inline(suggestion)
            else:
                self.chat_view.hide_autocomplete_inline()
        except asyncio.CancelledError:
            pass
        except Exception as e:
            print(f"Autocomplete error: {e}")
            self.chat_view.hide_autocomplete_inline()
