from PyQt6.QtCore import QEasingCurve, QPropertyAnimation, Qt
from PyQt6.QtGui import QPixmap
from PyQt6.QtWidgets import QApplication, QSplashScreen

from ui.common.BaseWindow import BaseWindow
from utils import resource_path
from utils.error_logger import log_info, log_warning


class WindowManager:
    __window: BaseWindow | None = None
    __splash: QSplashScreen | None = None
    __splash_animation: QPropertyAnimation | None = None
    __SPLASH_FADE_DURATION_MS = 250

    @staticmethod
    def show_splash(application: QApplication):
        splash_path = resource_path('assets/splash/splash_v1.png')
        if not splash_path:
            log_warning('Splash image path is empty. Splash will be skipped.')
            return

        pixmap = QPixmap(splash_path)
        if pixmap.isNull():
            log_warning(f'Failed to load splash pixmap from {splash_path}')
            return

        scaled_width = max(1, pixmap.width() // 2)
        scaled_height = max(1, pixmap.height() // 2)
        scaled_pixmap = pixmap.scaled(
            scaled_width,
            scaled_height,
            Qt.AspectRatioMode.KeepAspectRatio,
            Qt.TransformationMode.SmoothTransformation,
        )

        splash = QSplashScreen(scaled_pixmap)
        splash.setWindowFlag(Qt.WindowType.WindowStaysOnTopHint, True)
        splash.setEnabled(False)
        splash.setWindowOpacity(1.0)

        splash.show()
        application.processEvents()

        log_info('Splash screen displayed via WindowManager.')
        WindowManager.__splash = splash

    @staticmethod
    def hide_splash_screen(animated: bool = True):
        if not WindowManager.__splash:
            return

        if not animated:
            WindowManager.__cleanup_splash()
            return

        if WindowManager.__splash_animation:
            return

        animation = QPropertyAnimation(WindowManager.__splash, b"windowOpacity")
        animation.setDuration(WindowManager.__SPLASH_FADE_DURATION_MS)
        animation.setStartValue(1.0)
        animation.setEndValue(0.0)
        animation.setEasingCurve(QEasingCurve.Type.InOutQuad)

        def _on_fade_finished():
            WindowManager.__cleanup_splash()
            animation.deleteLater()
            WindowManager.__splash_animation = None

        animation.finished.connect(_on_fade_finished)
        animation.start()
        WindowManager.__splash_animation = animation

    @staticmethod
    def open_chat_window():
        print('MainWindow open_chat_window')
        from core.formChat.FormChat import FormChat
        chat = FormChat()
        WindowManager.__set_scene(chat)

    @staticmethod
    def open_login_window():
        print('MainWindow open_login_window')

        from core.FormMain import FormMain
        login = FormMain()
        WindowManager.__set_scene(login)

    @staticmethod
    def __set_scene(scene):
        if WindowManager.__window:
            WindowManager.__window.closeScene()
        scene.show()
        WindowManager.__window = scene
        WindowManager.hide_splash_screen()

    @staticmethod
    def __cleanup_splash():
        if WindowManager.__splash_animation:
            WindowManager.__splash_animation.stop()
            WindowManager.__splash_animation.deleteLater()
            WindowManager.__splash_animation = None

        if WindowManager.__splash:
            WindowManager.__splash.close()
            WindowManager.__splash.deleteLater()

        WindowManager.__splash = None
