import asyncio

from api import client
from ui.components.StandardDialog import StandardDialog
from ui.projectMembers.FormProjectMembers import UIFormProjectMembers
from core.I18n import _


class FormProjectMembers(UIFormProjectMembers):
    def __init__(self, parent, project_id=None):
        super().__init__(parent, project_id)
        self.project_id = project_id  # Ensure project_id prop is present
        self.invite_button.clicked.connect(self._on_invite_click)
        self.users_list.removeClicked.connect(self._on_email_clicked)
        asyncio.ensure_future(self._refresh_users_list())

    async def _refresh_users_list(self):
        self.users_list.clear()
        try:
            users = await client.project_user_list(self.project_id)
            self.users_list.addUsers(users)
            self.invite_input.setEnabled(True)
            self.invite_button.setEnabled(True)
        except Exception as e:
            StandardDialog.warn(self, _("Failed to fetch project user list."), str(e))
            self.invite_input.setEnabled(False)
            self.invite_button.setEnabled(False)

    def _on_invite_click(self):
        email = self.invite_input.text().strip()
        if not email:
            StandardDialog.warn(self, _("Please enter email to invite user to project"))
            return
        if '@' not in email:
            StandardDialog.warn(self, _("Invalid email format"))
            return
        asyncio.ensure_future(self._invite_user(email))

    async def _invite_user(self, email):
        self.invite_button.setLoading(True)
        try:
            await client.project_user_add(self.project_id, email)
            await self._refresh_users_list()
            StandardDialog.info(self, _("User %(email)s has been invited or added to project") % email)
            self.invite_input.clear()
        except Exception as e:
            StandardDialog.warn(self, _("Unable to invite user %(email)s") % email, str(e))
        finally:
            self.invite_button.setLoading(False)
            self.invite_input.setEnabled(True)

    def _on_email_clicked(self, user):
        email = user.email
        reply = StandardDialog.ask_remove(self, _("Are you sure to remove “%(email)s” from the project?") % email)
        if reply:
            asyncio.ensure_future(self._remove_user(email))

    async def _remove_user(self, email):
        try:
            await client.project_user_remove(self.project_id, email)
            await self._refresh_users_list()
            StandardDialog.info(self, _("Invitation for %(email)s has been removed from the project.") % email)
        except Exception as e:
            StandardDialog.warn(self, _("Unable to remove user %(email)s") % email, str(e))
