import asyncio

from PyQt6.QtCore import Qt

from api.client import send_otp, validate_otp
from core.I18n import _
from core.settings_manager import SettingsManager
from models.config.AppConfigModel import AppConfigModel
from ui.FormMain import Ui_FormMain


class FormMain(Ui_FormMain):

    def __init__(self):
        super().__init__()
        self.submit_button.clicked.connect(self.handle_submit_wrapper)
        self.session_id = None
        self.email = None
        self.otp_input.setVisible(False)  # Hide OTP input initially
        # self.chat_window = None  # Reference to chat window, if any
        # Register this as current login window
        self.config = AppConfigModel.load()
        self.email_input.setText(self.config.email)

        # --- Load app settings for always_on_top and opacity ---
        self.__apply_window_settings()

    def __apply_window_settings(self):
        settings = SettingsManager().instance().app_settings
        always_on_top = settings.always_on_top
        opacity = settings.opacity

        flags = self.windowFlags() & ~Qt.WindowType.WindowStaysOnTopHint
        if always_on_top:
            flags = flags | Qt.WindowType.WindowStaysOnTopHint
        self.setWindowFlags(flags)
        self.setWindowOpacity(opacity)
        self.show()  # Needed after setWindowFlags

    def handle_submit_wrapper(self):
        asyncio.create_task(self.handle_submit())

    async def handle_submit(self):
        if not self.email:
            # Send OTP
            self.email = self.email_input.text()
            self.config.email = self.email
            self.config.save()
            try:
                self.submit_button.setLoading(True)
                response = await send_otp(self.email)
                self.submit_button.setLoading(False)
                self.error_label.setText(_('OTP sent! Please check your email.'))
                self.email_input.setVisible(False)
                self.submit_button.setText(_('Confirm OTP'))
                self.otp_input.setVisible(True)
                self.otp_input.setFocus()
                self.otp_input.setText(response.get('data', {}).get('otp', ''))
            except Exception as e:
                self.error_label.setText(str(e))
                self.submit_button.setLoading(False)
                self.email = None
        else:
            # Validate OTP
            otp = self.otp_input.text()
            try:
                self.submit_button.setLoading(True)
                response, session_id = await validate_otp(self.email, otp)
                self.submit_button.setLoading(False)
                # Get session_id from cookie
                self.session_id = session_id
                self.config.session_id = session_id
                self.config.save()
                self.reset_ui()
                # Reset email BEFORE opening chat (ensures branch works next cycle)
                self.email = None

                from core.WindowManager import WindowManager
                WindowManager.open_chat_window()
                return  # Don't fall through after opening chat
            except Exception as e:
                self.error_label.setText(str(e))
                self.submit_button.setLoading(False)
