import asyncio

import api.client as client
from app_config import initialize_config
from core.I18n import _
from core.settings_manager import SettingsManager
from services.ApplicationService import ApplicationService
from ui.FormAppSettings import UIFormAppSettings
from ui.components.StandardDialog import StandardDialog


class FormAppSettings(UIFormAppSettings):
    def __init__(self, parent):
        super().__init__(parent)
        self.logout_button.clicked.connect(self.logout_wrapper)

        # Load settings
        self.load_settings()

        # Connect signals for sound settings
        self.checkbox_enable_sound.stateChanged.connect(self.save_settings)
        self.slider_volume.sliderReleased.connect(self.save_settings)
        # Connect signal for autocomplete checkbox
        self.checkbox_enable_autocomplete.stateChanged.connect(self.save_settings)
        # Connect signals for window settings (always on top and opacity)
        self.checkbox_always_on_top.stateChanged.connect(self.save_settings)
        self.slider_opacity.sliderReleased.connect(self.save_settings)

        # Display application version
        app_config = initialize_config()
        self.set_app_version(app_config.app_version)

        # get profile data
        asyncio.create_task(self.load_profile())

    def load_settings(self):
        settings = SettingsManager.instance().app_settings

        self.checkbox_enable_sound.setChecked(settings.enable_sound)
        self.slider_volume.setValue(settings.volume)
        self.checkbox_enable_autocomplete.setChecked(settings.enable_autocomplete)
        self.checkbox_always_on_top.setChecked(settings.always_on_top)
        self.slider_opacity.setValue(settings.opacity)
        self.window_group.setText(f"{_('Opacity')}: {settings.opacity}%")

    def save_settings(self):
        print("FormAppSettings.save_settings")
        settings = SettingsManager.instance().app_settings
        settings.enable_sound = self.checkbox_enable_sound.isChecked()
        settings.volume = self.slider_volume.value()
        settings.enable_autocomplete = self.checkbox_enable_autocomplete.isChecked()
        settings.always_on_top = self.checkbox_always_on_top.isChecked()
        settings.opacity = self.slider_opacity.value()
        SettingsManager.instance().save_settings()

    async def load_profile(self):
        try:
            profile_data = await client.profile()
            email = profile_data.get('email')
            plan = profile_data.get('plan', {}).get('name', '')
            glyph_balance = profile_data.get('glyph_balance', None)
            self.set_profile_data(email, plan, glyph_balance)
        except Exception as e:
            print(e)

    def show(self):
        self.exec()

    def logout_wrapper(self):
        asyncio.create_task(self.logout())

    async def logout(self):
        error = None
        try:
            await client.logout()
        except Exception as e:
            error = str(e)
        finally:
            try:
                if error:
                    StandardDialog.warn(self, _("Logout failed"), error)
            except Exception as diag_exc:
                print(f"Dialog error (non-critical): {diag_exc}")
            try:
                ApplicationService.logout()
            except Exception as w_exc:
                print(f"Main window close error (non-critical): {w_exc}")
